@extends('layouts.app')

@section('title', 'Post New Ad - Yilick')

@section('content')
<div class="bg-gray-50 min-h-screen py-8">
    <div class="container mx-auto px-4 max-w-3xl">
        
        <!-- Header -->
        <div class="mb-8 flex flex-col sm:flex-row sm:items-center justify-between gap-4">
            <div>
                <h1 class="text-3xl font-extrabold text-gray-900 tracking-tight">Post a New Ad</h1>
                <p class="text-gray-500 text-sm mt-1 flex items-center gap-2">
                    <span class="w-2 h-2 rounded-full bg-emerald-500"></span>
                    Reach thousands of potential buyers instantly.
                </p>
            </div>
            <a href="{{ route('web.my-ads') }}" class="inline-flex items-center justify-center gap-2 px-6 py-2.5 rounded-xl border border-gray-200 bg-white text-gray-600 font-semibold text-sm hover:bg-gray-50 hover:border-gray-300 hover:text-gray-900 transition-all shadow-sm">
                <i class="fa-solid fa-arrow-left text-xs"></i>
                Back to My Ads
            </a>
        </div>

        <form action="{{ route('listing.store') }}" method="POST" enctype="multipart/form-data" id="postAdForm" class="space-y-6">
            @csrf
            
            <!-- SECTION 1: Category & Photos -->
            <div class="bg-white rounded-2xl shadow-sm border border-gray-100 overflow-hidden">
                <div class="p-6 space-y-6">
                    <h2 class="text-lg font-bold text-gray-900 flex items-center gap-2">
                        <span class="flex items-center justify-center w-6 h-6 rounded-full bg-emerald-100 text-emerald-600 text-xs shadow-sm">1</span>
                        Category & Images
                    </h2>

                    <!-- Category Selection -->
                    <div class="grid grid-cols-1 md:grid-cols-2 gap-5">
                        <div class="space-y-2">
                            <label class="text-sm font-bold text-gray-700 ml-1">Category *</label>
                            <div class="relative group">
                                <select id="category" name="category" class="w-full rounded-2xl border-gray-200 bg-gray-50/50 focus:bg-white focus:border-emerald-500 focus:ring-4 focus:ring-emerald-500/10 transition-all py-3.5 px-4 text-sm font-medium appearance-none outline-none" required onchange="handleCategoryChange()">
                                    <option value="">Select Category</option>
                                </select>
                                <i class="fa-solid fa-chevron-down absolute right-4 top-1/2 -translate-y-1/2 text-gray-400 pointer-events-none text-xs"></i>
                            </div>
                        </div>

                        <div id="subCategoryContainer" class="hidden space-y-2">
                            <label class="text-sm font-bold text-gray-700 ml-1">Sub Category *</label>
                            <div class="relative group">
                                <select id="sub_category" name="sub_category" class="w-full rounded-2xl border-gray-200 bg-gray-50/50 focus:bg-white focus:border-emerald-500 focus:ring-4 focus:ring-emerald-500/10 transition-all py-3.5 px-4 text-sm font-medium appearance-none outline-none" required onchange="handleSubCategoryChange()">
                                    <option value="">Select Sub Category</option>
                                </select>
                                <i class="fa-solid fa-chevron-down absolute right-4 top-1/2 -translate-y-1/2 text-gray-400 pointer-events-none text-xs"></i>
                            </div>
                        </div>
                    </div>

                    <hr class="border-gray-100">

                    <!-- Photos -->
                    <div class="space-y-2">
                        <div class="flex items-center justify-between">
                            <label class="text-sm font-semibold text-gray-700">Photos *</label>
                            <span class="text-xs text-emerald-600 bg-emerald-50 px-2 py-0.5 rounded-full font-medium">First photo is cover</span>
                        </div>
                        
                        <div class="grid grid-cols-2 sm:grid-cols-4 gap-4" id="photo-grid">
                            <label id="add-photo-label" class="relative aspect-square rounded-2xl border-2 border-dashed border-gray-200 hover:border-primary hover:bg-primary/5 transition-all cursor-pointer group flex flex-col items-center justify-center gap-2">
                                <input type="file" multiple accept="image/*" class="absolute inset-0 opacity-0 cursor-pointer" onchange="handleFileSelect(this)">
                                <div class="w-12 h-12 rounded-full bg-gray-50 group-hover:bg-primary/10 flex items-center justify-center transition-colors">
                                    <i class="fa-solid fa-plus text-gray-400 group-hover:text-primary"></i>
                                </div>
                                <span class="text-xs font-bold text-gray-500 group-hover:text-primary">Add Photo</span>
                            </label>
                        </div>
                        <p class="text-xs text-gray-400">Supported formats: JPG, PNG, WEBP. Max size: 5MB per image.</p>
                    </div>
                </div>
            </div>

            <!-- SECTION 2: Item Details -->
            <div class="bg-white rounded-2xl shadow-sm border border-gray-100 overflow-hidden">
                <div class="p-6 space-y-6">
                    <h2 class="text-lg font-bold text-gray-900 flex items-center gap-2">
                        <span class="flex items-center justify-center w-6 h-6 rounded-full bg-emerald-100 text-emerald-600 text-xs shadow-sm">2</span>
                        Item Details
                    </h2>

                    <div class="space-y-5">
                        <div class="space-y-2">
                            <label class="text-sm font-bold text-gray-700 ml-1">Title *</label>
                            <div class="relative group">
                                <i class="fa-solid fa-tag absolute left-4 top-1/2 -translate-y-1/2 text-gray-400 group-focus-within:text-emerald-500 transition-colors"></i>
                                <input type="text" name="title" class="w-full rounded-2xl border-gray-200 bg-gray-50/50 focus:bg-white focus:border-emerald-500 focus:ring-4 focus:ring-emerald-500/10 transition-all py-3.5 pl-11 pr-4 text-sm font-medium placeholder:text-gray-400 outline-none" placeholder="e.g. Brand New iPhone 14 Pro Max 256GB" required minlength="5" maxlength="100">
                            </div>
                        </div>

                        <!-- Custom Fields Container -->
                        <div id="customFieldsContainer" class="grid grid-cols-1 md:grid-cols-2 gap-5"></div>

                        <div class="space-y-2">
                            <label class="text-sm font-bold text-gray-700 ml-1">Description *</label>
                            <textarea name="description" rows="6" class="w-full rounded-2xl border-gray-200 bg-gray-50/50 focus:bg-white focus:border-emerald-500 focus:ring-4 focus:ring-emerald-500/10 transition-all py-3.5 px-4 text-sm font-medium placeholder:text-gray-400 outline-none resize-none" placeholder="Describe the condition, features, and reason for selling. Be detailed to build trust with buyers..." required minlength="20"></textarea>
                        </div>

                        <div class="space-y-2">
                            <label class="text-sm font-bold text-gray-700 ml-1">Location *</label>
                            <div class="relative group">
                                <i class="fa-solid fa-location-dot absolute left-4 top-1/2 -translate-y-1/2 text-gray-400 group-focus-within:text-emerald-500 transition-colors"></i>
                                <select id="location" name="location" class="w-full rounded-2xl border-gray-200 bg-gray-50/50 focus:bg-white focus:border-emerald-500 focus:ring-4 focus:ring-emerald-500/10 transition-all py-3.5 pl-11 pr-4 text-sm font-medium appearance-none outline-none" required>
                                    <option value="">Select District</option>
                                </select>
                                <i class="fa-solid fa-chevron-down absolute right-4 top-1/2 -translate-y-1/2 text-gray-400 pointer-events-none text-xs"></i>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- SECTION 3: Price & Inventory -->
            <div class="bg-white rounded-2xl shadow-sm border border-gray-100 overflow-hidden">
                <div class="p-6 space-y-6">
                    <h2 class="text-lg font-bold text-gray-900 flex items-center gap-2">
                        <span class="flex items-center justify-center w-6 h-6 rounded-full bg-emerald-100 text-emerald-600 text-xs shadow-sm">3</span>
                        Pricing & Stock
                    </h2>

                    <div class="space-y-2">
                        <label class="text-sm font-bold text-gray-700 ml-1">Selling Price *</label>
                        <div class="relative group">
                            <span class="absolute left-5 top-1/2 -translate-y-1/2 text-gray-900 font-extrabold text-sm tracking-wide">UGX</span>
                            <input type="text" name="price" class="w-full rounded-2xl border-gray-200 bg-gray-50/50 focus:bg-white focus:border-emerald-500 focus:ring-4 focus:ring-emerald-500/10 transition-all py-4 pl-16 pr-4 font-extrabold text-2xl text-gray-900 price-input placeholder:text-gray-300 outline-none shadow-inner" required placeholder="0">
                        </div>
                    </div>

                    <div class="bg-gray-50 rounded-xl p-4 border border-gray-100 space-y-4">
                        <div class="flex items-center justify-between">
                            <div class="flex items-center gap-2">
                                <div class="bg-blue-100 text-blue-600 p-2 rounded-lg">
                                    <i class="fa-solid fa-cash-register text-sm"></i>
                                </div>
                                <div>
                                    <span class="text-sm font-bold text-gray-900 block">Stock Management</span>
                                    <span class="text-xs text-gray-500">Track stock and calculate profit (Private)</span>
                                </div>
                            </div>
                            <label class="relative inline-flex items-center cursor-pointer">
                                <input type="checkbox" name="allow_pos" id="allowPosToggle" class="sr-only peer" onchange="toggleStockFields()">
                                <div class="w-11 h-6 bg-gray-200 peer-focus:outline-none peer-focus:ring-4 peer-focus:ring-emerald-300 rounded-full peer peer-checked:after:translate-x-full rtl:peer-checked:after:-translate-x-full peer-checked:after:border-white after:content-[''] after:absolute after:top-[2px] after:start-[2px] after:bg-white after:border-gray-300 after:border after:rounded-full after:h-5 after:w-5 after:transition-all peer-checked:bg-emerald-600"></div>
                            </label>
                        </div>

                        <div id="stockFields" class="hidden pt-3 border-t border-gray-100 mt-3 animate-fade-in">
                            <div class="bg-emerald-50/50 border border-emerald-100 rounded-xl p-4 flex items-start gap-4 shadow-sm">
                                <div class="w-10 h-10 rounded-full bg-emerald-100/50 flex items-center justify-center text-emerald-600 shrink-0">
                                    <i class="fa-solid fa-mobile-screen-button"></i>
                                </div>
                                <div class="space-y-1">
                                    <p class="text-sm font-bold text-gray-900">App-Managed Inventory</p>
                                    <p class="text-xs text-gray-600 leading-relaxed">
                                        You can manage stock quantities, and record restocks via the <strong>Point of Sale (POS)</strong> tool within the <strong>Business Tools</strong> section of the Yilick app.
                                    </p>
                                </div>
                            </div>
                        </div>
                    </div>
            </div>

            <!-- SECTION 4: Contact Info -->
            <div class="bg-white rounded-2xl shadow-sm border border-gray-100 overflow-hidden">
                <div class="p-4 bg-gray-50 border-b border-gray-100 flex items-center justify-between">
                    <h2 class="text-sm font-bold text-gray-700">Contact Information</h2>
                    <span class="text-xs text-gray-400"><i class="fa-solid fa-lock mr-1"></i>Secure</span>
                </div>
                <div class="p-6 grid grid-cols-1 md:grid-cols-2 gap-4">
                     <div class="space-y-2">
                         <label class="text-xs font-bold text-gray-400 uppercase tracking-widest ml-1">Contact Name</label>
                         <div class="relative group">
                            <i class="fa-regular fa-user absolute left-4 top-1/2 -translate-y-1/2 text-gray-400 group-focus-within:text-emerald-500 transition-colors"></i>
                            <input type="text" value="{{ $business->name ?? auth()->user()->full_name }}" class="w-full rounded-2xl border-gray-200 bg-gray-100/50 text-gray-500 font-bold text-sm py-3.5 pl-11 pr-4 cursor-not-allowed border-dashed" readonly>
                         </div>
                    </div>
                     <div class="space-y-2">
                         <label class="text-xs font-bold text-gray-400 uppercase tracking-widest ml-1">Contact Phone</label>
                         <div class="relative group">
                            <i class="fa-solid fa-phone absolute left-4 top-1/2 -translate-y-1/2 text-gray-400 group-focus-within:text-emerald-500 transition-colors"></i>
                            <input type="text" value="{{ $business->primary_phone ?? auth()->user()->phone }}" class="w-full rounded-2xl border-gray-200 bg-gray-100/50 text-gray-500 font-bold text-sm py-3.5 pl-11 pr-4 cursor-not-allowed border-dashed" readonly>
                         </div>
                    </div>
                </div>
            </div>

            <!-- Submit -->
            <div class="pt-8 pb-16">
                <button type="submit" class="w-full md:w-auto md:min-w-[400px] block mx-auto py-5 px-10 bg-emerald-600 text-white font-black rounded-2xl shadow-xl shadow-emerald-200 hover:bg-emerald-700 hover:shadow-emerald-300 hover:-translate-y-1 transition-all transform active:scale-95 text-xl flex items-center justify-center gap-3 group">
                    <span>Post Ad Now</span>
                    <i class="fa-solid fa-rocket group-hover:translate-x-1 group-hover:-translate-y-1 transition-transform"></i>
                </button>
                <div class="flex flex-col items-center gap-2 mt-6">
                    <p class="text-center text-xs text-gray-400 max-w-sm">
                        By clicking "Post Ad Now", you agree to Yilick's <a href="{{ route('web.terms') }}" class="text-emerald-600 hover:underline font-bold">Terms of Service</a> and <a href="{{ route('web.privacy') }}" class="text-emerald-600 hover:underline font-bold">Privacy Policy</a>.
                    </p>
                    <div class="flex items-center gap-4 py-2 opacity-50">
                        <i class="fa-brands fa-cc-visa text-xl"></i>
                        <i class="fa-brands fa-cc-mastercard text-xl"></i>
                        <i class="fa-solid fa-shield-halved text-lg"></i>
                    </div>
                </div>
            </div>
        </form>
    </div>
</div>

<!-- Submission Overlay -->
<div id="submissionOverlay" class="fixed inset-0 z-[200] hidden opacity-0 transition-opacity duration-500 bg-white/80 backdrop-blur-md flex flex-col items-center justify-center">
    <div class="relative">
        <!-- Pulse Rings -->
        <div class="absolute inset-0 scale-[3] animate-ping opacity-20 bg-emerald-400 rounded-full"></div>
        <div class="absolute inset-0 scale-[2] animate-ping opacity-30 bg-emerald-300 rounded-full" style="animation-delay: 0.5s"></div>
        
        <div class="relative bg-white p-8 rounded-3xl shadow-2xl flex flex-col items-center gap-6 border border-emerald-50">
            <div class="w-20 h-20 rounded-2xl bg-emerald-50 text-emerald-600 flex items-center justify-center overflow-hidden">
                <i class="fa-solid fa-cloud-arrow-up text-4xl animate-bounce"></i>
            </div>
            <div class="text-center space-y-2">
                <h3 class="text-2xl font-black text-gray-900 tracking-tight" id="submissionOverlayTitle">Posting your ad...</h3>
                <p class="text-gray-500 text-sm font-medium">Please wait while we prepare your listing</p>
            </div>
            
            <!-- Progress Bar -->
            <div class="w-64 h-2 bg-gray-100 rounded-full overflow-hidden relative">
                <div class="absolute inset-0 bg-emerald-500 w-1/3 animate-[progress_2s_infinite_linear]"></div>
            </div>
        </div>
    </div>
</div>

<style>
@keyframes progress {
    0% { left: -100%; width: 50%; }
    50% { left: 50%; width: 80%; }
    100% { left: 100%; width: 50%; }
}
</style>

<!-- Checkbox Multi-Select Modal -->
<div id="checkboxModal" class="fixed inset-0 z-[100] hidden opacity-0 transition-opacity duration-300" aria-labelledby="modal-title" role="dialog" aria-modal="true">
    <div class="fixed inset-0 bg-gray-900/60 backdrop-blur-sm transition-opacity" onclick="closeCheckboxModal()"></div>
    <div class="flex min-h-full items-end sm:items-center justify-center p-0 sm:p-4 text-center">
        <div class="relative transform overflow-hidden rounded-t-3xl sm:rounded-2xl bg-white text-left shadow-2xl transition-all w-full sm:max-w-lg scale-95 duration-300 translate-y-10 sm:translate-y-0" id="checkboxModalPanel">
            <div class="bg-white px-4 pb-4 pt-5 sm:p-6">
                <div class="flex items-center justify-between mb-4">
                    <h3 class="text-xl font-bold text-gray-900" id="checkboxModalTitle">Select Options</h3>
                    <button type="button" onclick="closeCheckboxModal()" class="text-gray-400 hover:text-gray-600 transition-colors">
                        <i class="fa-solid fa-xmark text-xl"></i>
                    </button>
                </div>
                
                <!-- Search -->
                <div class="relative mb-4">
                    <i class="fa-solid fa-magnifying-glass absolute left-3 top-1/2 -translate-y-1/2 text-gray-400 text-sm"></i>
                    <input type="text" id="checkboxSearch" placeholder="Search options..." class="w-full rounded-xl border-gray-200 bg-gray-50 pl-10 py-2.5 text-sm focus:bg-white focus:ring-emerald-500 focus:border-emerald-500 transition-all">
                </div>

                <!-- Options List -->
                <div id="checkboxOptionsList" class="max-h-[60vh] overflow-y-auto space-y-1 pr-2 custom-scrollbar">
                    <!-- Options will be injected here -->
                </div>
            </div>
            <div class="bg-gray-50 px-4 py-4 sm:flex sm:flex-row-reverse sm:px-6 border-t border-gray-100">
                <button type="button" onclick="confirmCheckboxSelection()" class="w-full inline-flex justify-center rounded-xl bg-emerald-600 px-6 py-3 text-sm font-bold text-white shadow-lg shadow-emerald-200 hover:bg-emerald-700 sm:ml-3 sm:w-auto transition-all">
                    Apply Selection
                </button>
                <button type="button" onclick="closeCheckboxModal()" class="mt-3 w-full inline-flex justify-center rounded-xl bg-white px-3 py-3 text-sm font-bold text-gray-700 shadow-sm ring-1 ring-inset ring-gray-300 hover:bg-gray-50 sm:mt-0 sm:w-auto transition-all">
                    Cancel
                </button>
            </div>
        </div>
    </div>
</div>

<style>
    .custom-scrollbar::-webkit-scrollbar { width: 4px; }
    .custom-scrollbar::-webkit-scrollbar-track { background: transparent; }
    .custom-scrollbar::-webkit-scrollbar-thumb { background: #E5E7EB; border-radius: 10px; }
    .custom-scrollbar::-webkit-scrollbar-thumb:hover { background: #D1D5DB; }
</style>

@push('scripts')
<script>
    // Data Storage
    let categoriesData = [];
    let typesData = {};
    let districtsData = [];
    let customFieldsMap = {}; // From types.json
    let currentCheckboxField = null; // Track which field is being edited in modal
    let tempCheckboxSelection = []; // Track temp selections in modal

    document.addEventListener('DOMContentLoaded', async () => {
        await Promise.all([
            fetchCategories(),
            fetchDistricts(),
            fetchTypes()
        ]);
    });

    // formatted number input
    document.querySelectorAll('.price-input').forEach(input => {
        input.addEventListener('input', function(e) {
            // Remove non-digits
            let value = this.value.replace(/\D/g, '');
            if (value === '') return;
            
            // Format with commas
            this.value = parseInt(value).toLocaleString();
        });
    });

    // Form Submit Handler
    document.getElementById('postAdForm').addEventListener('submit', function(e) {
        const form = this;
        const mediaInputs = Array.from(document.querySelectorAll('input[name="media_ids[]"]'));
        const hasUploadedPhoto = mediaInputs.some(input => input.value !== '');
        
        // 1. Photo Validation
        if (!hasUploadedPhoto) {
            e.preventDefault();
            const photoGrid = document.getElementById('photo-grid');
            photoGrid.classList.add('ring-2', 'ring-red-500', 'ring-offset-2');
            
            // Show error message (or update existing one)
            let errorMsg = document.getElementById('photo-error-message');
            if (!errorMsg) {
                errorMsg = document.createElement('p');
                errorMsg.id = 'photo-error-message';
                errorMsg.className = 'text-red-500 text-xs font-bold mt-2 animate-bounce';
                errorMsg.innerHTML = '<i class="fa-solid fa-circle-exclamation mr-1"></i> Please upload at least one photo to continue.';
                photoGrid.parentNode.appendChild(errorMsg);
            }
            
            photoGrid.scrollIntoView({ behavior: 'smooth', block: 'center' });
            return;
        }

        // 2. Standard Field Validation (HTML5)
        if (!form.checkValidity()) {
            e.preventDefault();
            form.reportValidity();
            return;
        }

        // 3. Preparation for submission (strip commas)
        document.querySelectorAll('.price-input').forEach(input => {
            input.value = input.value.replace(/,/g, '');
        });
        
        // Disable submit button to prevent double clicks
        const submitBtn = form.querySelector('button[type="submit"]');
        submitBtn.disabled = true;
        submitBtn.innerHTML = '<i class="fa-solid fa-circle-notch fa-spin mr-2"></i> Processing...';
    });

    async function fetchCategories() {
        try {
            const response = await fetch('/assets/categories.json');
            const data = await response.json();
            // Filter where parent is 0
            categoriesData = data;
            const parents = data.filter(item => item.parent === '0');
            
            const select = document.getElementById('category');
            parents.forEach(cat => {
                const opt = document.createElement('option');
                opt.value = cat.Category; 
                opt.dataset.id = cat.id;
                opt.textContent = cat.Category;
                select.appendChild(opt);
            });
        } catch (e) {
            console.error('Failed to load categories', e);
        }
    }

    async function fetchDistricts() {
        try {
            const response = await fetch('/assets/districts.json');
            const data = await response.json();
            const select = document.getElementById('location');
            data.forEach(d => {
                const opt = document.createElement('option');
                opt.value = d.name;
                opt.textContent = d.name;
                select.appendChild(opt);
            });

            // Pre-select business district if available
            const businessDistrict = "{{ $business->district ?? '' }}";
            if (businessDistrict) {
                select.value = businessDistrict;
            }
        } catch (e) {
            console.error('Failed to load districts', e);
        }
    }

    async function fetchTypes() {
        try {
            const response = await fetch('/assets/types.json');
            const data = await response.json();
            customFieldsMap = data.customFields || {};
        } catch (e) {
            console.error('Failed to load types', e);
        }
    }

    function handleCategoryChange() {
        const catSelect = document.getElementById('category');
        const selectedId = catSelect.options[catSelect.selectedIndex].dataset.id;
        
        // Reset Sub
        const subSelect = document.getElementById('sub_category');
        subSelect.innerHTML = '<option value="">Select Sub Category</option>';
        document.getElementById('subCategoryContainer').classList.add('hidden');
        document.getElementById('customFieldsContainer').innerHTML = '';

        if (!selectedId) return;

        const subs = categoriesData.filter(item => item.parent === selectedId);
        
        if (subs.length > 0) {
            subs.forEach(sub => {
                const opt = document.createElement('option');
                opt.value = sub.Category;
                opt.dataset.id = sub.id;
                opt.textContent = sub.Category;
                subSelect.appendChild(opt);
            });
            document.getElementById('subCategoryContainer').classList.remove('hidden');
        }
    }

    function handleSubCategoryChange() {
        const subSelect = document.getElementById('sub_category');
        const selectedId = subSelect.options[subSelect.selectedIndex].dataset.id;

        // Render Custom Fields for this Sub Category
        renderCustomFields(selectedId);
    }

    function renderCustomFields(subCategoryId) {
        const container = document.getElementById('customFieldsContainer');
        container.innerHTML = '';
        
        const fields = customFieldsMap[subCategoryId] || [];
        
        fields.forEach(field => {
            const wrapper = document.createElement('div');
            wrapper.className = 'space-y-1';

            const label = document.createElement('label');
            label.className = 'text-sm font-bold text-gray-700 ml-1';
            label.textContent = field.name + (field.isRequired ? ' *' : '');
            
            let input;
            const inputClasses = 'w-full rounded-2xl border-gray-200 bg-gray-50/50 focus:bg-white focus:border-emerald-500 focus:ring-4 focus:ring-emerald-500/10 transition-all py-3.5 px-4 text-sm font-medium placeholder:text-gray-400 outline-none';

            if (field.type === 'dropdown' && field.options) {
                input = document.createElement('select');
                input.className = inputClasses;
                const empty = document.createElement('option');
                empty.text = `Select ${field.name}`;
                empty.value = '';
                input.add(empty);
                field.options.forEach(opt => {
                    const o = document.createElement('option');
                    o.value = opt;
                    o.text = opt;
                    input.add(o);
                });
            } else if (field.type === 'checkbox' && field.options) {
                // Multi-select Modal Trigger
                const triggerId = `checkbox_trigger_${field.name.replace(/\s+/g, '_')}`;
                const hiddenInputId = `checkbox_hidden_${field.name.replace(/\s+/g, '_')}`;
                
                // Display Area
                input = document.createElement('div');
                input.id = triggerId;
                input.className = `${inputClasses} cursor-pointer flex items-center justify-between min-h-[46px] group`;
                input.innerHTML = `
                    <span class="text-gray-400 group-hover:text-gray-600 transition-colors">Select ${field.name}</span>
                    <i class="fa-solid fa-chevron-right text-xs text-gray-400 group-hover:translate-x-0.5 transition-all"></i>
                `;
                input.onclick = () => openCheckboxModal(field);

                // Hidden Inputs Container for form submission
                const hiddenContainer = document.createElement('div');
                hiddenContainer.id = hiddenInputId;
                wrapper.appendChild(hiddenContainer);

                // No direct input variable here, we use the trigger
                wrapper.appendChild(label);
                wrapper.appendChild(input);
                container.appendChild(wrapper);
                return; // Early return as we handled it specially
            } else if (field.type === 'textarea') {
                input = document.createElement('textarea');
                input.rows = 3;
                input.className = inputClasses;
            } else if (field.type === 'number') {
                input = document.createElement('input');
                input.type = 'number';
                input.className = inputClasses;
            } else {
                // Text
                input = document.createElement('input');
                input.type = 'text';
                input.className = inputClasses;
            }

            input.name = `custom_fields[${field.name}]`;
            if (field.isRequired) input.required = true;

            wrapper.appendChild(label);
            wrapper.appendChild(input);
            container.appendChild(wrapper);
        });
    }
    function toggleStockFields() {
        const checked = document.getElementById('allowPosToggle').checked;
        const container = document.getElementById('stockFields');
        if (checked) {
            container.classList.remove('hidden');
            container.classList.add('block');
        } else {
            container.classList.add('hidden');
            container.classList.remove('block');
        }
    }

    let mediaIds = [];
    let uploadQueue = 0;

    function handleFileSelect(input) {
        // Clear photo error state if it exists
        const photoGrid = document.getElementById('photo-grid');
        photoGrid.classList.remove('ring-2', 'ring-red-500', 'ring-offset-2');
        const errorMsg = document.getElementById('photo-error-message');
        if (errorMsg) errorMsg.remove();

        const files = Array.from(input.files);
        const grid = document.getElementById('photo-grid');
        const existingPhotos = grid.querySelectorAll('.photo-preview').length;
        const remainingSlots = 8 - existingPhotos;
        const filesToProcess = files.slice(0, remainingSlots);

        filesToProcess.forEach(file => {
            uploadFile(file);
        });
        
        // Clear input so same file can be selected again
        input.value = '';
    }

    function uploadFile(file) {
        const grid = document.getElementById('photo-grid');
        const label = document.getElementById('add-photo-label');
        const tempId = 'temp-' + Date.now() + '-' + Math.random().toString(36).substr(2, 9);
        
        // 1. Create Preview Item
        const div = document.createElement('div');
        div.id = tempId;
        div.className = 'photo-preview relative aspect-square rounded-2xl overflow-hidden group shadow-sm border border-gray-100 bg-gray-50';
        div.innerHTML = `
            <div class="absolute inset-0 flex items-center justify-center bg-gray-50">
                <div class="text-primary animate-spin">
                    <i class="fa-solid fa-circle-notch fa-2xl"></i>
                </div>
            </div>
            <img class="w-full h-full object-cover opacity-0 transition-opacity duration-300">
            <div class="absolute inset-0 bg-black/40 opacity-0 group-hover:opacity-100 transition-opacity flex items-center justify-center gap-2">
                <button type="button" onclick="removePhoto('${tempId}')" class="w-8 h-8 rounded-full bg-white text-red-500 flex items-center justify-center hover:bg-red-50 hover:text-red-600 transition-colors shadow-lg">
                    <i class="fa-solid fa-trash-can"></i>
                </button>
            </div>
            <div class="upload-progress absolute bottom-0 left-0 h-1 bg-primary transition-all duration-300" style="width: 0%"></div>
            <input type="hidden" name="media_ids[]" value="">
        `;
        
        grid.insertBefore(div, label);
        updateCoverBadge();

        // 2. Set Local Preview
        const reader = new FileReader();
        reader.onload = (e) => {
            const img = div.querySelector('img');
            img.src = e.target.result;
            img.classList.remove('opacity-0');
            div.querySelector('.absolute.inset-0.flex').classList.add('hidden');
        };
        reader.readAsDataURL(file);

        // 3. AJAX Upload
        const formData = new FormData();
        formData.append('media[]', file);
        formData.append('_token', '{{ csrf_token() }}');

        uploadQueue++;
        checkSubmitStatus();

        const xhr = new XMLHttpRequest();
        xhr.open('POST', '{{ route('listing.media.upload') }}', true);
        
        xhr.upload.onprogress = (e) => {
            if (e.lengthComputable) {
                const percent = (e.loaded / e.total) * 100;
                div.querySelector('.upload-progress').style.width = percent + '%';
            }
        };

        xhr.onload = function() {
            uploadQueue--;
            checkSubmitStatus();
            
            if (xhr.status === 200) {
                const response = JSON.parse(xhr.responseText);
                if (response.success && response.media.length > 0) {
                    const media = response.media[0];
                    div.querySelector('input[name="media_ids[]"]').value = media.id;
                    div.querySelector('.upload-progress').classList.add('bg-emerald-500');
                    setTimeout(() => div.querySelector('.upload-progress').style.opacity = '0', 1000);
                } else {
                    handleUploadError(tempId);
                }
            } else {
                handleUploadError(tempId);
            }
        };

        xhr.onerror = function() {
            uploadQueue--;
            checkSubmitStatus();
            handleUploadError(tempId);
        };

        xhr.send(formData);
    }

    function handleUploadError(tempId) {
        const div = document.getElementById(tempId);
        if (!div) return;
        
        div.classList.add('border-red-500');
        div.innerHTML += `
            <div class="absolute inset-0 bg-red-500/10 flex items-center justify-center pointer-events-none">
                <div class="bg-red-500 text-white px-2 py-1 rounded-lg text-[10px] font-bold">UPLOAD FAILED</div>
            </div>
        `;
        // Flash the trash button to encourage removal
        const trashBtn = div.querySelector('button');
        trashBtn.classList.remove('bg-white');
        trashBtn.classList.add('bg-red-500', 'text-white');
    }

    function removePhoto(id) {
        const element = document.getElementById(id);
        if (element) {
            element.remove();
            updateCoverBadge();
            checkSubmitStatus();
        }
    }

    function updateCoverBadge() {
        const grid = document.getElementById('photo-grid');
        const previews = grid.querySelectorAll('.photo-preview');
        
        previews.forEach((preview, index) => {
            // Remove existing badge
            const oldBadge = preview.querySelector('.cover-badge');
            if (oldBadge) oldBadge.remove();

            // Add to first one
            if (index === 0) {
                const badge = document.createElement('div');
                badge.className = 'cover-badge absolute top-2 left-2 px-2 py-0.5 rounded-md bg-primary text-white text-[10px] font-bold shadow-sm z-10';
                badge.textContent = 'COVER';
                preview.appendChild(badge);
            }
        });
    }

    function checkSubmitStatus() {
        const submitBtn = document.querySelector('button[type="submit"]');
        const previews = document.querySelectorAll('.photo-preview');
        const hasUploading = uploadQueue > 0;
        const hasPhotos = previews.length > 0;
        
        // We can also check if any have failed uploads (empty hidden input)
        let hasFailed = false;
        previews.forEach(p => {
            if (!p.querySelector('input[name="media_ids[]"]').value && !hasUploading) {
                hasFailed = true;
            }
        });

        if (hasUploading) {
            submitBtn.disabled = true;
            submitBtn.classList.add('opacity-75', 'cursor-not-allowed');
            submitBtn.innerHTML = '<i class="fa-solid fa-circle-notch fa-spin mr-2"></i>Uploading Photos...';
        } else {
            submitBtn.disabled = false;
            submitBtn.classList.remove('opacity-75', 'cursor-not-allowed');
            submitBtn.innerHTML = 'Post Ad Now';
        }
    }

    // Prevent submission during upload and show overlay
    document.getElementById('postAdForm').addEventListener('submit', function(e) {
        if (uploadQueue > 0) {
            e.preventDefault();
            alert('Please wait for photos to finish uploading before posting.');
            return;
        }

        // Show Overlay
        const overlay = document.getElementById('submissionOverlay');
        overlay.classList.remove('hidden');
        setTimeout(() => overlay.classList.remove('opacity-0'), 10);
        
        // Disable button further
        const submitBtn = this.querySelector('button[type="submit"]');
        submitBtn.disabled = true;
    });

    // --- Checkbox Modal Logic ---
    function openCheckboxModal(field) {
        currentCheckboxField = field;
        const modal = document.getElementById('checkboxModal');
        const panel = document.getElementById('checkboxModalPanel');
        const title = document.getElementById('checkboxModalTitle');
        const list = document.getElementById('checkboxOptionsList');
        const searchInput = document.getElementById('checkboxSearch');

        title.textContent = `Select ${field.name}`;
        searchInput.value = '';
        list.innerHTML = '';

        // Get current selection from hidden inputs
        const currentSelection = Array.from(document.querySelectorAll(`input[name="custom_fields[${field.name}][]"]`))
            .map(input => input.value);
        tempCheckboxSelection = [...currentSelection];

        // Populate options
        field.options.forEach(opt => {
            const isChecked = tempCheckboxSelection.includes(opt);
            const item = document.createElement('label');
            item.className = 'flex items-center justify-between p-3 rounded-xl hover:bg-emerald-50 cursor-pointer transition-colors group option-item';
            item.innerHTML = `
                <span class="text-sm font-medium text-gray-700 group-hover:text-emerald-700">${opt}</span>
                <input type="checkbox" value="${opt}" ${isChecked ? 'checked' : ''} onchange="toggleTempSelection(this)" 
                    class="w-5 h-5 rounded text-emerald-600 focus:ring-emerald-500 border-gray-300">
            `;
            list.appendChild(item);
        });

        // Search logic
        searchInput.oninput = (e) => {
            const query = e.target.value.toLowerCase();
            document.querySelectorAll('.option-item').forEach(item => {
                const text = item.textContent.toLowerCase();
                item.style.display = text.includes(query) ? 'flex' : 'none';
            });
        };

        // Show modal
        modal.classList.remove('hidden');
        setTimeout(() => {
            modal.classList.remove('opacity-0');
            panel.classList.remove('scale-95', 'translate-y-10');
            panel.classList.add('scale-100', 'translate-y-0');
        }, 10);
    }

    function toggleTempSelection(cb) {
        if (cb.checked) {
            if (!tempCheckboxSelection.includes(cb.value)) tempCheckboxSelection.push(cb.value);
        } else {
            tempCheckboxSelection = tempCheckboxSelection.filter(v => v !== cb.value);
        }
    }

    function closeCheckboxModal() {
        const modal = document.getElementById('checkboxModal');
        const panel = document.getElementById('checkboxModalPanel');
        
        modal.classList.add('opacity-0');
        panel.classList.add('scale-95', 'translate-y-10');
        panel.classList.remove('scale-100', 'translate-y-0');
        
        setTimeout(() => {
            modal.classList.add('hidden');
        }, 300);
    }

    function confirmCheckboxSelection() {
        if (!currentCheckboxField) return;

        const fieldName = currentCheckboxField.name;
        const triggerId = `checkbox_trigger_${fieldName.replace(/\s+/g, '_')}`;
        const hiddenId = `checkbox_hidden_${fieldName.replace(/\s+/g, '_')}`;
        
        const trigger = document.getElementById(triggerId);
        const hiddenContainer = document.getElementById(hiddenId);

        // Update trigger display
        if (tempCheckboxSelection.length > 0) {
            trigger.innerHTML = `
                <div class="flex flex-wrap gap-1 py-1">
                    ${tempCheckboxSelection.slice(0, 3).map(v => `<span class="bg-emerald-100 text-emerald-700 text-[10px] font-bold px-2 py-0.5 rounded-lg">${v}</span>`).join('')}
                    ${tempCheckboxSelection.length > 3 ? `<span class="text-gray-400 text-[10px] font-bold">+${tempCheckboxSelection.length - 3} more</span>` : ''}
                </div>
                <i class="fa-solid fa-chevron-right text-xs text-emerald-600"></i>
            `;
        } else {
            trigger.innerHTML = `
                <span class="text-gray-400">Select ${fieldName}</span>
                <i class="fa-solid fa-chevron-right text-xs text-gray-400"></i>
            `;
        }

        // Update hidden inputs
        hiddenContainer.innerHTML = '';
        tempCheckboxSelection.forEach(val => {
            const input = document.createElement('input');
            input.type = 'hidden';
            input.name = `custom_fields[${fieldName}][]`;
            input.value = val;
            hiddenContainer.appendChild(input);
        });

        closeCheckboxModal();
    }
</script>
@endpush
@endsection
