@extends('layouts.app')

@section('title', 'Messages')

@push('styles')
<style>
    .chat-bg {
        background-color: #f0f2f5;
        position: relative;
    }
    
    /* Elegant Split Background */
    .chat-bg::before {
        content: "";
        position: absolute;
        top: 0;
        left: 0;
        right: 0;
        height: 220px;
        background-color: #00cc50;
        z-index: 0;
    }

    .main-container {
        position: relative;
        z-index: 10;
    }
    
    .segmented-control {
        background: #f0f2f5;
        padding: 4px;
        border-radius: 14px;
        display: flex;
        position: relative;
    }

    .segmented-control button {
        position: relative;
        z-index: 1;
        transition: color 0.3s ease;
    }

    .segmented-control .active-bg {
        position: absolute;
        top: 4px;
        bottom: 4px;
        width: calc(50% - 4px);
        background: white;
        border-radius: 10px;
        box-shadow: 0 2px 8px rgba(0,0,0,0.06);
        transition: transform 0.4s cubic-bezier(0.18, 0.89, 0.32, 1.28);
        z-index: 0;
    }

    .tab-selling-active .active-bg {
        transform: translateX(100%);
    }

    .conversation-card {
        transition: all 0.2s ease;
    }

    .conversation-card:hover {
        background-color: #f8fafc;
        transform: translateY(-1px);
    }

    .unread-indicator {
        width: 12px;
        height: 12px;
        background-color: #00cc50;
        border-radius: 50%;
        box-shadow: 0 0 0 3px white;
    }
    
    .search-input-wrapper {
        background: #f6f6f6;
        border-radius: 12px;
        transition: all 0.2s ease;
    }

    .search-input-wrapper:focus-within {
        background: white;
        box-shadow: 0 0 0 2px rgba(0, 204, 80, 0.2);
    }

    /* Skeleton Loader */
    @keyframes shimmer {
        0% { background-position: -200% 0; }
        100% { background-position: 200% 0; }
    }
    .skeleton {
        background: linear-gradient(90deg, #f1f5f9 25%, #e2e8f0 50%, #f1f5f9 75%);
        background-size: 200% 100%;
        animation: shimmer 1.5s infinite;
    }

    /* Responsive adjustments */
    @media (max-width: 768px) {
        .chat-bg::before {
            height: 120px;
        }
    }
</style>
@endpush

@section('content')
<div class="chat-bg min-h-screen font-sans pb-12 md:pb-20">
    <div class="main-container container mx-auto px-0 md:px-4 max-w-4xl py-6 md:py-12">
        
        <div class="flex flex-col md:flex-row md:items-end justify-between px-6 md:px-0 mb-6 md:mb-10 gap-4">
            <div>
                <h1 class="text-3xl md:text-5xl font-black text-white md:text-white tracking-tight drop-shadow-sm">Messages</h1>
                <p class="text-white/80 text-sm mt-1 font-medium hidden md:block">Seamlessly connect with your marketplace community</p>
            </div>
            
            <!-- Mobile Header Only: Brief Info -->
            <div class="md:hidden flex items-center justify-between mt-2">
                <span id="mobile-stats" class="text-white/90 text-sm font-bold">Inbox Overview</span>
            </div>
        </div>

        <!-- Professional Overhaul Container -->
        <div class="bg-white md:rounded-[2.5rem] shadow-2xl shadow-black/10 border border-white/20 overflow-hidden flex flex-col min-h-[700px] relative">
            
            <!-- Tools Header (Search & Tabs) -->
            <div class="p-4 md:p-6 border-b border-gray-100 bg-white/50 backdrop-blur-xl sticky top-0 z-30 space-y-4">
                <!-- Tabs -->
                <div id="tab-container" class="segmented-control max-w-sm mx-auto">
                    <div class="active-bg"></div>
                    <button onclick="switchTab('buying')" id="tab-buying" 
                            class="flex-1 py-2.5 text-sm font-black transition-all text-primary">
                        <i class="fa-solid fa-basket-shopping mr-2 text-[12px]"></i>Buying
                    </button>
                    <button onclick="switchTab('selling')" id="tab-selling" 
                            class="flex-1 py-2.5 text-sm font-black transition-all text-gray-500 hover:text-gray-700">
                        <i class="fa-solid fa-tags mr-2 text-[12px]"></i>Selling
                    </button>
                </div>

                <!-- Search Bar -->
                <div class="search-input-wrapper flex items-center px-4 py-2.5 max-w-2xl mx-auto shadow-sm">
                    <i class="fa-solid fa-magnifying-glass text-gray-400 text-sm mr-3"></i>
                    <input type="text" 
                           id="conversation-search"
                           placeholder="Search conversations or products..." 
                           oninput="filterConversations(this.value)"
                           class="w-full bg-transparent border-none focus:ring-0 text-sm font-medium text-gray-800 placeholder-gray-400">
                </div>
            </div>

            <!-- List Overlay Container -->
            <div class="flex-1 flex flex-col relative min-h-0 bg-white">
                <!-- Loader -->
                <div id="loader" class="flex flex-col divide-y divide-gray-50">
                    @for ($i = 0; $i < 6; $i++)
                    <div class="flex items-center gap-4 p-5">
                        <div class="w-14 h-14 rounded-full skeleton flex-shrink-0"></div>
                        <div class="flex-1 space-y-3">
                            <div class="flex justify-between">
                                <div class="h-4 w-1/3 skeleton rounded-lg"></div>
                                <div class="h-3 w-16 skeleton rounded-lg"></div>
                            </div>
                            <div class="h-3 w-3/4 skeleton rounded-lg"></div>
                        </div>
                    </div>
                    @endfor
                </div>

                <!-- Empty State -->
                <div id="empty-state" class="hidden absolute inset-0 text-center flex flex-col items-center justify-center p-12 bg-white z-10 animate-in fade-in zoom-in duration-500">
                    <div class="w-28 h-28 bg-[#f0f2f5] rounded-full flex items-center justify-center mx-auto mb-6 shadow-inner">
                        <i class="fa-regular fa-comments text-4xl text-gray-300"></i>
                    </div>
                    <h3 class="text-2xl font-black text-gray-900 mb-2">Clean Slate</h3>
                    <p class="text-gray-500 max-w-xs mx-auto text-sm leading-relaxed font-medium" id="empty-text">
                        Select a listing to start a conversation.
                    </p>
                </div>

                <!-- The Actual List -->
                <div id="conversation-list" class="divide-y divide-gray-50 flex flex-col bg-white overflow-y-auto" style="scrollbar-width: thin; max-height: calc(100vh - 400px);">
                    <!-- Conversations injected here -->
                </div>
            </div>
        </div>
    </div>
</div>

@push('scripts')
<script>
    const currentUserId = {{ auth()->id() ?? 0 }};
    let currentTab = 'buying';
    let pollingInterval = null;
    let allConversations = []; // Store for filtering

    document.addEventListener('DOMContentLoaded', () => {
        loadConversations();
        startPolling();
    });

    function switchTab(tab) {
        if (currentTab === tab) return;
        currentTab = tab;
        
        const container = document.getElementById('tab-container');
        const btnBuying = document.getElementById('tab-buying');
        const btnSelling = document.getElementById('tab-selling');
        const searchInput = document.getElementById('conversation-search');

        searchInput.value = ''; // Reset search on tab switch

        if (tab === 'buying') {
            container.classList.remove('tab-selling-active');
            btnBuying.classList.add('text-primary');
            btnBuying.classList.remove('text-gray-500');
            btnSelling.classList.remove('text-primary');
            btnSelling.classList.add('text-gray-500');
            document.getElementById('empty-text').innerText = 'Select a listing to start a conversation.';
        } else {
            container.classList.add('tab-selling-active');
            btnSelling.classList.add('text-primary');
            btnSelling.classList.remove('text-gray-500');
            btnBuying.classList.remove('text-primary');
            btnBuying.classList.add('text-gray-500');
            document.getElementById('empty-text').innerText = 'Messages from potential buyers will appear here.';
        }

        document.getElementById('conversation-list').innerHTML = '';
        document.getElementById('loader').classList.remove('hidden');
        document.getElementById('empty-state').classList.add('hidden');
        loadConversations();
    }

    function loadConversations() {
        axios.get(`{{ route('web.chat.conversations') }}?type=${currentTab}`)
            .then(response => {
                allConversations = response.data.conversations;
                renderConversations(allConversations);
            })
            .catch(error => {
                console.error('Error loading conversations:', error);
                document.getElementById('loader').classList.add('hidden');
            });
    }

    function filterConversations(query) {
        const q = query.toLowerCase().trim();
        if (!q) {
            renderConversations(allConversations);
            return;
        }

        const filtered = allConversations.filter(c => 
            (c.name && c.name.toLowerCase().includes(q)) || 
            (c.listing_title && c.listing_title.toLowerCase().includes(q)) ||
            (c.last_message && c.last_message.toLowerCase().includes(q))
        );
        renderConversations(filtered);
    }

    function renderConversations(conversations) {
        const list = document.getElementById('conversation-list');
        const emptyState = document.getElementById('empty-state');
        const loader = document.getElementById('loader');
        
        loader.classList.add('hidden');

        if (conversations.length === 0) {
            emptyState.classList.remove('hidden');
            list.innerHTML = '';
            return;
        }

        emptyState.classList.add('hidden');
        
        let html = '';
        conversations.forEach(c => {
            const isUnread = c.unread_count > 0;
            const avatar = c.logo ? c.logo : '{{ asset('assets/icons/favicon.png') }}';
            const listingImg = c.listing_image ? c.listing_image : '{{ asset('assets/icons/listing_placeholder.svg') }}';
            
            html += `
            <a href="${c.url}" class="conversation-card group flex items-start gap-4 p-5 transition-all duration-200 border-l-[6px] ${isUnread ? 'bg-primary/5 border-l-primary' : 'border-l-transparent'}">
                <!-- User Avatar -->
                <div class="relative flex-shrink-0">
                    <img src="${avatar}" onerror="this.src='{{ asset('assets/icons/favicon.png') }}'" class="w-16 h-16 rounded-full object-cover border-4 border-white shadow-md ring-1 ring-gray-100/50">
                    ${isUnread ? '<div class="absolute top-1 -right-1 unread-indicator animate-pulse"></div>' : ''}
                </div>

                <!-- Chat Info -->
                <div class="flex-1 min-w-0 py-1">
                    <div class="flex justify-between items-center mb-1.5">
                        <div class="flex items-center gap-2 min-w-0">
                            <h4 class="text-[17px] font-black text-gray-800 truncate leading-tight group-hover:text-primary transition-colors">${c.name}</h4>
                            ${c.is_verified ? '<i class="fa-solid fa-circle-check text-blue-500 text-[13px]"></i>' : ''}
                        </div>
                        <div class="flex flex-col items-end">
                            <span class="text-[10px] text-gray-400 font-bold whitespace-nowrap bg-gray-100/50 px-2 py-1 rounded-full uppercase tracking-tight">${c.last_message_at}</span>
                            ${isUnread ? `<span class="mt-1 bg-primary text-white text-[10px] font-black px-1.5 py-0.5 rounded-full min-w-[20px] text-center shadow-sm shadow-primary/30">${c.unread_count}</span>` : ''}
                        </div>
                    </div>
                    
                    <div class="flex items-center gap-2 mb-2.5">
                        <span class="text-[9px] uppercase tracking-tighter font-black text-primary bg-primary/10 px-2 py-0.5 rounded-md">Listing</span>
                        <p class="text-[12px] text-gray-600 font-extrabold truncate">${c.listing_title}</p>
                    </div>

                    <div class="flex items-center gap-1.5">
                        ${c.last_message_sender_id == currentUserId ? '<i class="fa-solid fa-check-double text-[11px] text-[#53bdeb]"></i>' : ''}
                        <p class="text-[14px] ${isUnread ? 'text-gray-900 font-black' : 'text-gray-500 font-medium'} truncate line-clamp-1 italic">
                            ${c.last_message ? c.last_message : '<span class="text-gray-300 font-normal">No messages yet</span>'}
                        </p>
                    </div>
                </div>

                <!-- Listing Thumbnail -->
                <div class="hidden sm:flex items-center gap-4 flex-shrink-0 ml-2">
                    <div class="w-16 h-16 rounded-[1.25rem] overflow-hidden border-2 border-white shadow-xl ring-1 ring-gray-100 transition-transform duration-500 group-hover:rotate-2 group-hover:scale-110">
                        <img src="${listingImg}" onerror="this.src='{{ asset('assets/icons/listing_placeholder.svg') }}'" class="w-full h-full object-cover">
                    </div>
                </div>
            </a>`;
        });

        list.innerHTML = html;
    }

    function startPolling() {
        if (pollingInterval) clearInterval(pollingInterval);
        pollingInterval = setInterval(() => {
            // Don't poll while user is searching to avoid losing focus/state
            if (document.getElementById('conversation-search').value.trim() !== '') return;

            axios.get(`{{ route('web.chat.conversations') }}?type=${currentTab}`)
                .then(response => {
                    allConversations = response.data.conversations;
                    renderConversations(allConversations);
                })
                .catch(console.error);
        }, 15000);
    }
</script>
@endpush
@endsection
