<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        Schema::create('users', function (Blueprint $table) {
            $table->id();
            $table->uuid('uuid')->unique();
            
            // Profile Information
            $table->string('profile_photo')->nullable();
            $table->string('full_name');
            $table->string('username')->unique()->nullable();
            $table->text('bio')->nullable();
            $table->date('date_of_birth')->nullable();
            $table->enum('gender', ['male', 'female', 'other', 'prefer_not_to_say'])->nullable();
            
            // Contact Information
            $table->string('email')->unique();
            $table->timestamp('email_verified_at')->nullable();
            $table->boolean('is_email_verified')->default(false);
            $table->string('phone', 20)->unique()->nullable();
            $table->timestamp('phone_verified_at')->nullable();
            $table->boolean('is_phone_verified')->default(false);
            $table->string('secondary_phone', 20)->nullable();
            $table->string('whatsapp_number', 20)->nullable();
            
            // Location Information
            $table->string('country', 100)->nullable();
            $table->string('country_code', 3)->nullable();
            $table->string('state', 100)->nullable();
            $table->string('city', 100)->nullable();
            $table->string('district', 100)->nullable();
            $table->string('address')->nullable();
            $table->string('postal_code', 20)->nullable();
            $table->decimal('latitude', 10, 8)->nullable();
            $table->decimal('longitude', 11, 8)->nullable();
            $table->string('timezone', 50)->nullable();
            
            // Social & Web Presence
            $table->string('website')->nullable();
            $table->string('facebook_url')->nullable();
            $table->string('twitter_url')->nullable();
            $table->string('instagram_url')->nullable();
            $table->string('linkedin_url')->nullable();
            $table->string('tiktok_url')->nullable();
            $table->string('youtube_url')->nullable();
            
            // Account Security
            $table->string('password');
            $table->boolean('two_factor_enabled')->default(false);
            $table->string('two_factor_secret')->nullable();
            $table->text('two_factor_recovery_codes')->nullable();
            $table->timestamp('password_changed_at')->nullable();
            $table->unsignedTinyInteger('failed_login_attempts')->default(0);
            $table->timestamp('locked_until')->nullable();
            $table->timestamp('last_login_at')->nullable();
            $table->string('last_login_ip', 45)->nullable();
            
            // Preferences & Settings
            $table->string('language', 10)->default('en');
            $table->string('currency', 3)->default('USD');
            $table->boolean('is_notifications_enabled')->default(true);
            $table->boolean('is_email_notifications_enabled')->default(true);
            $table->boolean('is_sms_notifications_enabled')->default(true);
            $table->boolean('is_push_notifications_enabled')->default(true);
            $table->boolean('is_marketing_emails_enabled')->default(false);
            $table->boolean('show_phone_publicly')->default(false);
            $table->boolean('show_email_publicly')->default(false);
            $table->boolean('show_location_publicly')->default(true);
            
            // Identity Verification
            $table->enum('identity_verification_status', ['unverified', 'pending', 'verified', 'rejected'])->default('unverified');
            $table->string('identity_document_type', 50)->nullable();
            $table->string('identity_document_number')->nullable();
            $table->string('identity_document_photo')->nullable();
            $table->timestamp('identity_verified_at')->nullable();
            
            // Account Status & Role
            $table->enum('status', ['active', 'suspended', 'banned', 'pending', 'deleted'])->default('active');
            $table->string('suspension_reason')->nullable();
            $table->timestamp('suspended_at')->nullable();
            $table->timestamp('suspension_ends_at')->nullable();
            $table->enum('role', ['user', 'seller', 'premium_seller', 'moderator', 'admin', 'super_admin'])->default('user');
            
            // Referral System
            $table->string('referral_code', 20)->unique()->nullable();
            $table->foreignId('referred_by')->nullable()->constrained('users')->nullOnDelete();
            $table->unsignedInteger('referral_count')->default(0);
            
            // Metadata
            $table->string('registration_source', 50)->nullable();
            $table->string('registration_ip', 45)->nullable();
            $table->string('device_type', 50)->nullable();
            $table->json('metadata')->nullable();
            
            $table->rememberToken();
            $table->timestamps();
            $table->softDeletes();

            // Indexes for performance
            $table->index(['status', 'role']);
            $table->index('created_at');
            $table->index(['country_code', 'city']);
            $table->index(['latitude', 'longitude']);
            $table->index('identity_verification_status');
            $table->index('referral_code');
        });

        Schema::create('password_reset_tokens', function (Blueprint $table) {
            $table->string('email')->primary();
            $table->string('token');
            $table->timestamp('created_at')->nullable();
        });

        Schema::create('sessions', function (Blueprint $table) {
            $table->string('id')->primary();
            $table->foreignId('user_id')->nullable()->index();
            $table->string('ip_address', 45)->nullable();
            $table->text('user_agent')->nullable();
            $table->longText('payload');
            $table->integer('last_activity')->index();
        });
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        Schema::dropIfExists('users');
        Schema::dropIfExists('password_reset_tokens');
        Schema::dropIfExists('sessions');
    }
};
