<?php

namespace App\Services;

use App\Models\SearchQuery;
use Illuminate\Http\Request;

class SearchTrackingService
{
    public function trackSearch(
        Request $request,
        string $searchType,
        ?string $query = null,
        ?array $filters = null,
        int $resultsCount = 0
    ): SearchQuery {
        return SearchQuery::create([
            'user_id' => auth('api')->id(),
            'search_type' => $searchType,
            'query' => $query,
            'filters' => $filters,
            'results_count' => $resultsCount,
            'ip_address' => $request->ip(),
            'user_agent' => $request->userAgent(),
            'session_id' => $this->getSessionId($request),
            'searched_at' => now(),
        ]);
    }

    protected function getSessionId(Request $request): ?string
    {
        try {
            return $request->session()->getId();
        } catch (\RuntimeException $e) {
            return md5($request->ip() . $request->userAgent());
        }
    }

    public function getPopularSearches(int $limit = 10, ?string $searchType = null)
    {
        $query = SearchQuery::selectRaw('query, search_type, COUNT(*) as search_count, MAX(searched_at) as last_searched')
            ->whereNotNull('query')
            ->groupBy('query', 'search_type')
            ->orderByDesc('search_count');

        if ($searchType) {
            $query->where('search_type', $searchType);
        }

        return $query->limit($limit)->get();
    }

    public function getUserSearchHistory(int $userId, int $limit = 50)
    {
        return SearchQuery::where('user_id', $userId)
            ->orderByDesc('searched_at')
            ->limit($limit)
            ->get();
    }

    public function getSearchAnalytics(?string $searchType = null, ?string $period = '7days')
    {
        $startDate = match($period) {
            '24hours' => now()->subDay(),
            '7days' => now()->subDays(7),
            '30days' => now()->subDays(30),
            '90days' => now()->subDays(90),
            default => now()->subDays(7),
        };

        $query = SearchQuery::where('searched_at', '>=', $startDate);

        if ($searchType) {
            $query->where('search_type', $searchType);
        }

        $totalSearches = $query->count();
        $uniqueUsers = $query->whereNotNull('user_id')->distinct('user_id')->count('user_id');
        $anonymousSearches = $query->whereNull('user_id')->count();
        $avgResultsCount = $query->avg('results_count');

        $searchesByType = SearchQuery::selectRaw('search_type, COUNT(*) as count')
            ->where('searched_at', '>=', $startDate)
            ->groupBy('search_type')
            ->get();

        $searchesByDay = SearchQuery::selectRaw('DATE(searched_at) as date, COUNT(*) as count')
            ->where('searched_at', '>=', $startDate)
            ->groupBy('date')
            ->orderBy('date')
            ->get();

        return [
            'period' => $period,
            'start_date' => $startDate->toDateTimeString(),
            'end_date' => now()->toDateTimeString(),
            'total_searches' => $totalSearches,
            'unique_users' => $uniqueUsers,
            'anonymous_searches' => $anonymousSearches,
            'authenticated_searches' => $totalSearches - $anonymousSearches,
            'avg_results_count' => round($avgResultsCount, 2),
            'searches_by_type' => $searchesByType,
            'searches_by_day' => $searchesByDay,
        ];
    }

    public function getZeroResultSearches(int $limit = 20, ?string $searchType = null)
    {
        $query = SearchQuery::where('results_count', 0)
            ->whereNotNull('query')
            ->orderByDesc('searched_at');

        if ($searchType) {
            $query->where('search_type', $searchType);
        }

        return $query->limit($limit)->get();
    }
}
