<?php

namespace App\Services;

use App\Models\Listing;
use App\Models\ListingMedia;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;

class ReverseImageSearchService
{
    protected ImageRecognitionService $recognitionService;
    
    public function __construct()
    {
        $this->recognitionService = new ImageRecognitionService();
    }

    /**
     * Search for similar listings based on uploaded image
     * Combines content hash matching and visual similarity
     * 
     * @param string $imagePath Path to uploaded search image
     * @param int $limit Maximum results to return
     * @return Collection Collection of listings with similarity scores
     */
    public function searchByImage(string $imagePath, int $limit = 20): Collection
    {
        // Step 1: Calculate content hash of search image
        $searchHash = $this->calculateHash($imagePath);
        
        // Step 2: Get visual features from Google Vision API
        $visualData = $this->recognitionService->detectWebEntities($imagePath);
        $searchEntities = collect($visualData['entities'])->pluck('description')->toArray();
        $searchLabels = $visualData['labels'];
        
        // Step 3: Find exact hash matches (highest priority)
        $exactMatches = $this->findExactMatches($searchHash);
        
        // Step 4: Find visually similar matches based on tags/labels
        $visualMatches = $this->findVisualMatches($searchEntities, $searchLabels);
        
        // Step 5: Merge and rank results
        $results = $this->mergeAndRankResults($exactMatches, $visualMatches, $limit);
        
        return $results;
    }

    /**
     * Calculate SHA-256 hash of image file
     */
    protected function calculateHash(string $imagePath): string
    {
        $fullPath = storage_path('app/public/' . $imagePath);
        return hash_file('sha256', $fullPath);
    }

    /**
     * Find listings with exact content hash match
     */
    protected function findExactMatches(string $hash): Collection
    {
        return Listing::query()
            ->publicVisible()
            ->whereHas('media', function ($query) use ($hash) {
                $query->where('content_hash', $hash);
            })
            ->with(['business', 'media'])
            ->get()
            ->map(function ($listing) {
                return [
                    'listing' => $listing,
                    'similarity_score' => 1.0, // Perfect match
                    'match_type' => 'exact_hash',
                ];
            });
    }

    /**
     * Find listings with similar visual content based on tags
     */
    protected function findVisualMatches(array $entities, array $labels): Collection
    {
        if (empty($entities) && empty($labels)) {
            return collect();
        }

        // Combine entities and labels for matching
        $searchTerms = array_merge($entities, $labels);
        $searchTerms = array_unique(array_filter($searchTerms));
        
        if (empty($searchTerms)) {
            return collect();
        }

        // Find listings with matching tags
        $listings = Listing::query()
            ->publicVisible()
            ->whereHas('media', function ($query) use ($searchTerms) {
                foreach ($searchTerms as $term) {
                    $query->orWhereJsonContains('tags', [['label' => $term]]);
                }
            })
            ->with(['business', 'media'])
            ->limit(100) // Get more for scoring
            ->get();

        // Calculate similarity scores based on tag overlap
        return $listings->map(function ($listing) use ($searchTerms) {
            $score = $this->calculateSimilarityScore($listing, $searchTerms);
            
            return [
                'listing' => $listing,
                'similarity_score' => $score,
                'match_type' => 'visual_similarity',
            ];
        })->filter(function ($item) {
            return $item['similarity_score'] > 0.3; // Minimum 30% similarity
        });
    }

    /**
     * Calculate similarity score based on tag overlap
     */
    protected function calculateSimilarityScore(Listing $listing, array $searchTerms): float
    {
        $allMediaTags = $listing->media->flatMap(function ($media) {
            return collect($media->tags ?? [])->pluck('label')->toArray();
        })->unique()->toArray();

        if (empty($allMediaTags) || empty($searchTerms)) {
            return 0.0;
        }

        // Calculate Jaccard similarity coefficient
        $intersection = count(array_intersect($searchTerms, $allMediaTags));
        $union = count(array_unique(array_merge($searchTerms, $allMediaTags)));

        $jaccardScore = $union > 0 ? $intersection / $union : 0;

        // Boost score if high-confidence tags match
        $confidenceBoost = 0;
        foreach ($listing->media as $media) {
            foreach ($media->tags ?? [] as $tag) {
                if (in_array($tag['label'], $searchTerms) && ($tag['confidence'] ?? 0) > 0.8) {
                    $confidenceBoost += 0.1;
                }
            }
        }

        return min(1.0, $jaccardScore + ($confidenceBoost * 0.1));
    }

    /**
     * Merge exact and visual matches, remove duplicates, and rank by score
     */
    protected function mergeAndRankResults(Collection $exactMatches, Collection $visualMatches, int $limit): Collection
    {
        // Merge results
        $allResults = $exactMatches->concat($visualMatches);

        // Remove duplicates (keep highest score)
        $uniqueResults = $allResults->groupBy(function ($item) {
            return $item['listing']->id;
        })->map(function ($group) {
            return $group->sortByDesc('similarity_score')->first();
        })->values();

        // Sort by similarity score and limit
        return $uniqueResults
            ->sortByDesc('similarity_score')
            ->take($limit)
            ->values();
    }
}
