<?php

namespace App\Services;

use App\Models\Listing;
use App\Models\ListingPromotion;
use App\Models\UserSubscription;
use Carbon\Carbon;

class PremiumService
{
    public function canBusinessCreateListing(int $businessId): array
    {
        $subscription = $this->getActiveSubscriptionForBusiness($businessId);

        if (!$subscription) {
            return [
                'allowed' => true,
                'reason' => 'No subscription - can create free listings',
                'subscription' => null,
            ];
        }

        if ($subscription->canCreateListing()) {
            return [
                'allowed' => true,
                'reason' => 'Within subscription limits',
                'subscription' => $subscription,
                'remaining' => $subscription->package->max_listings - $subscription->listings_used,
            ];
        }

        return [
            'allowed' => false,
            'reason' => 'Subscription listing limit reached',
            'subscription' => $subscription,
            'limit' => $subscription->package->max_listings,
            'used' => $subscription->listings_used,
        ];
    }

    public function incrementListingUsage(int $businessId, int $listingId): void
    {
        $subscription = $this->getActiveSubscriptionForBusiness($businessId);

        if ($subscription) {
            Listing::where('id', $listingId)->update([
                'subscription_id' => $subscription->id,
            ]);
        }
    }

    public function featureListing(int $listingId, int $userId, ?int $businessId = null, ?array $paymentData = null): array
    {
        $listing = Listing::findOrFail($listingId);
        $subscription = $businessId ? $this->getActiveSubscriptionForBusiness($businessId) : $this->getActiveSubscription($userId);

        if ($subscription && $subscription->canFeatureListing()) {
            $subscription->increment('featured_listings_used');
            
            $listing->update([
                'is_featured' => true,
                'featured_until' => now()->addDays(30),
                'subscription_id' => $subscription->id,
            ]);

            return [
                'success' => true,
                'method' => 'subscription',
                'message' => 'Listing featured using subscription quota',
            ];
        }

        if ($paymentData) {
            $promotion = ListingPromotion::create([
                'listing_id' => $listingId,
                'user_id' => $userId,
                'type' => 'feature',
                'amount_paid' => $paymentData['amount'],
                'payment_method' => $paymentData['payment_method'],
                'transaction_id' => $paymentData['transaction_id'],
                'starts_at' => now(),
                'expires_at' => now()->addDays($paymentData['duration_days'] ?? 30),
                'status' => 'active',
            ]);

            $listing->update([
                'is_featured' => true,
                'featured_until' => $promotion->expires_at,
            ]);

            return [
                'success' => true,
                'method' => 'individual_payment',
                'message' => 'Listing featured with individual payment',
                'promotion' => $promotion,
            ];
        }

        return [
            'success' => false,
            'message' => 'No active subscription quota and no payment provided',
        ];
    }

    public function bumpListing(int $listingId, int $userId, ?int $businessId = null, ?array $paymentData = null): array
    {
        $listing = Listing::findOrFail($listingId);
        $subscription = $businessId ? $this->getActiveSubscriptionForBusiness($businessId) : $this->getActiveSubscription($userId);

        if ($subscription && $subscription->canBumpListing()) {
            $subscription->increment('bumps_used_this_month');
            
            $listing->update([
                'bumped_at' => now(),
            ]);

            return [
                'success' => true,
                'method' => 'subscription',
                'message' => 'Listing bumped using subscription quota',
                'bumps_remaining' => $subscription->package->max_bumps_per_month - $subscription->bumps_used_this_month,
            ];
        }

        if ($paymentData) {
            $promotion = ListingPromotion::create([
                'listing_id' => $listingId,
                'user_id' => $userId,
                'type' => 'bump',
                'amount_paid' => $paymentData['amount'],
                'payment_method' => $paymentData['payment_method'],
                'transaction_id' => $paymentData['transaction_id'],
                'starts_at' => now(),
                'status' => 'active',
            ]);

            $listing->update([
                'bumped_at' => now(),
            ]);

            return [
                'success' => true,
                'method' => 'individual_payment',
                'message' => 'Listing bumped with individual payment',
                'promotion' => $promotion,
            ];
        }

        return [
            'success' => false,
            'message' => 'No active subscription quota and no payment provided',
        ];
    }


    public function processExpiredPromotions(): void
    {
        $expiredPromotions = ListingPromotion::where('status', 'active')
            ->where('expires_at', '<=', now())
            ->get();

        foreach ($expiredPromotions as $promotion) {
            $promotion->update(['status' => 'expired']);

            if ($promotion->type === 'feature') {
                Listing::where('id', $promotion->listing_id)->update([
                    'is_featured' => false,
                    'featured_until' => null,
                ]);
            }
        }
    }

    public function getActiveSubscription(int $userId): ?UserSubscription
    {
        return UserSubscription::where('user_id', $userId)
            ->whereNull('business_id') // Legacy or user-level
            ->where('status', 'active')
            ->where('expires_at', '>', now())
            ->with('package')
            ->first();
    }

    public function getActiveSubscriptionForBusiness(int $businessId): ?UserSubscription
    {
        return UserSubscription::where('business_id', $businessId)
            ->where('status', 'active')
            ->where('expires_at', '>', now())
            ->with('package')
            ->first();
    }

    public function getUserPromotions(int $userId, ?string $type = null)
    {
        $query = ListingPromotion::where('user_id', $userId)
            ->with('listing');

        if ($type) {
            $query->where('type', $type);
        }

        return $query->latest()->get();
    }
}
