<?php

namespace App\Services;

use App\Models\Listing;
use App\Models\ListingView;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class ListingViewTrackerService
{
    /**
     * Track a view for a listing
     * Prevents duplicate views within 24 hours from same user/session
     * 
     * @param Listing $listing
     * @param Request $request
     * @return bool True if view was recorded, false if duplicate
     */
    public function trackView(Listing $listing, Request $request): bool
    {
        $userId = auth('api')->id();
        $ipAddress = $request->ip();
        $userAgent = $request->userAgent();
        $sessionId = $this->getSessionId($request);

        // Check if this is a duplicate view within 24 hours
        $isDuplicate = $this->isDuplicateView($listing->id, $userId, $sessionId, $ipAddress);

        if ($isDuplicate) {
            return false;
        }

        // Record the view
        DB::transaction(function () use ($listing, $userId, $ipAddress, $userAgent, $sessionId) {
            // Create view record
            ListingView::create([
                'listing_id' => $listing->id,
                'user_id' => $userId,
                'ip_address' => $ipAddress,
                'user_agent' => $userAgent,
                'session_id' => $sessionId,
                'viewed_at' => now(),
            ]);

            // Increment total views count
            $listing->increment('views_count');

            // Update unique views count if this is first view from this user/session
            $isFirstView = $this->isFirstViewFromViewer($listing->id, $userId, $sessionId);
            if ($isFirstView) {
                $listing->increment('unique_views_count');
            }
        });

        return true;
    }

    /**
     * Get session ID safely (works with or without session middleware)
     */
    protected function getSessionId(Request $request): ?string
    {
        try {
            return $request->session()->getId();
        } catch (\RuntimeException $e) {
            // Session not available in stateless API context
            // Generate a pseudo-session ID from IP + User Agent for tracking
            return md5($request->ip() . $request->userAgent());
        }
    }

    /**
     * Check if this is a duplicate view within 24 hours
     */
    protected function isDuplicateView(int $listingId, ?int $userId, ?string $sessionId, ?string $ipAddress): bool
    {
        $query = ListingView::where('listing_id', $listingId)
            ->where('viewed_at', '>', now()->subHours(24));

        // Check by user ID if authenticated
        if ($userId) {
            $query->where('user_id', $userId);
        } 
        // Otherwise check by session or IP
        else {
            $query->where(function ($q) use ($sessionId, $ipAddress) {
                if ($sessionId) {
                    $q->where('session_id', $sessionId);
                }
                if ($ipAddress) {
                    $q->orWhere('ip_address', $ipAddress);
                }
            });
        }

        return $query->exists();
    }

    /**
     * Check if this is the first view ever from this viewer
     */
    protected function isFirstViewFromViewer(int $listingId, ?int $userId, ?string $sessionId): bool
    {
        $query = ListingView::where('listing_id', $listingId)
            ->where('viewed_at', '<', now()->subHours(24)); // Exclude current view

        if ($userId) {
            $query->where('user_id', $userId);
        } else if ($sessionId) {
            $query->where('session_id', $sessionId);
        }

        return !$query->exists();
    }

    /**
     * Get view statistics for a listing
     */
    public function getViewStats(Listing $listing): array
    {
        return [
            'total_views' => $listing->views_count,
            'unique_views' => $listing->unique_views_count,
            'views_today' => ListingView::where('listing_id', $listing->id)
                ->where('viewed_at', '>', now()->subDay())
                ->count(),
            'views_this_week' => ListingView::where('listing_id', $listing->id)
                ->where('viewed_at', '>', now()->subWeek())
                ->count(),
            'views_this_month' => ListingView::where('listing_id', $listing->id)
                ->where('viewed_at', '>', now()->subMonth())
                ->count(),
        ];
    }

    /**
     * Get list of viewers for a listing
     */
    public function getViewers(Listing $listing, int $limit = 50): array
    {
        $views = ListingView::where('listing_id', $listing->id)
            ->with('user:id,first_name,last_name,email')
            ->orderBy('viewed_at', 'desc')
            ->limit($limit)
            ->get();

        return $views->map(function ($view) {
            return [
                'id' => $view->id,
                'user' => $view->user ? [
                    'id' => $view->user->id,
                    'name' => trim($view->user->first_name . ' ' . $view->user->last_name),
                    'email' => $view->user->email,
                ] : null,
                'ip_address' => $view->user ? null : $this->maskIpAddress($view->ip_address),
                'viewed_at' => $view->viewed_at->toIso8601String(),
                'is_authenticated' => $view->user_id !== null,
            ];
        })->toArray();
    }

    /**
     * Mask IP address for privacy (show only first 2 octets)
     */
    protected function maskIpAddress(?string $ip): ?string
    {
        if (!$ip) {
            return null;
        }

        $parts = explode('.', $ip);
        if (count($parts) === 4) {
            return $parts[0] . '.' . $parts[1] . '.xxx.xxx';
        }

        return 'xxx.xxx.xxx.xxx';
    }
}
