<?php

namespace App\Services;

use App\Models\Listing;
use App\Models\ListingClick;
use App\Models\ListingImpression;
use App\Models\ListingInteraction;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class ListingAnalyticsService
{
    /**
     * Track impression (listing shown in feed/search)
     * Call this from frontend when listing appears in viewport
     */
    public function trackImpression(
        int $listingId,
        ?string $source = null,
        ?array $context = null,
        ?Request $request = null
    ): bool {
        $request = $request ?? request();
        
        // Prevent duplicate impressions within 1 hour
        $isDuplicate = $this->isDuplicateImpression($listingId, $request);
        if ($isDuplicate) {
            return false;
        }

        DB::transaction(function () use ($listingId, $source, $context, $request) {
            ListingImpression::create([
                'listing_id' => $listingId,
                'user_id' => auth('api')->id(),
                'source' => $source,
                'ip_address' => $request->ip(),
                'session_id' => $this->getSessionId($request),
                'context' => $context,
                'impressed_at' => now(),
            ]);

            Listing::where('id', $listingId)->increment('impressions_count');
        });

        return true;
    }

    /**
     * Track click (user clicked listing from feed/search)
     * Call this from frontend when user clicks listing card
     */
    public function trackClick(
        int $listingId,
        ?string $source = null,
        ?array $context = null,
        ?Request $request = null
    ): bool {
        $request = $request ?? request();

        DB::transaction(function () use ($listingId, $source, $context, $request) {
            ListingClick::create([
                'listing_id' => $listingId,
                'user_id' => auth('api')->id(),
                'source' => $source,
                'ip_address' => $request->ip(),
                'session_id' => $this->getSessionId($request),
                'context' => $context,
                'clicked_at' => now(),
            ]);

            Listing::where('id', $listingId)->increment('clicks_count');
        });

        return true;
    }

    /**
     * Track interaction (phone reveal, WhatsApp click, share, etc.)
     */
    public function trackInteraction(
        int $listingId,
        string $interactionType,
        ?array $metadata = null,
        ?Request $request = null
    ): bool {
        $request = $request ?? request();

        DB::transaction(function () use ($listingId, $interactionType, $metadata, $request) {
            ListingInteraction::create([
                'listing_id' => $listingId,
                'user_id' => auth('api')->id(),
                'interaction_type' => $interactionType,
                'ip_address' => $request->ip(),
                'session_id' => $this->getSessionId($request),
                'metadata' => $metadata,
                'interacted_at' => now(),
            ]);

            // Increment specific counters
            $this->incrementInteractionCounter($listingId, $interactionType);
        });

        return true;
    }

    /**
     * Get session ID safely (works with or without session middleware)
     */
    protected function getSessionId(Request $request): ?string
    {
        try {
            return $request->session()->getId();
        } catch (\RuntimeException $e) {
            // Session not available in stateless API context
            // Generate a pseudo-session ID from IP + User Agent for tracking
            return md5($request->ip() . $request->userAgent());
        }
    }

    /**
     * Check if impression is duplicate within 1 hour
     */
    protected function isDuplicateImpression(int $listingId, Request $request): bool
    {
        $userId = auth('api')->id();
        $sessionId = $this->getSessionId($request);
        $ipAddress = $request->ip();

        $query = ListingImpression::where('listing_id', $listingId)
            ->where('impressed_at', '>', now()->subHour());

        if ($userId) {
            // For authenticated users, check by user_id
            $query->where('user_id', $userId);
        } else {
            // For anonymous users, check by session_id OR IP address
            $query->where(function ($q) use ($sessionId, $ipAddress) {
                if ($sessionId) {
                    $q->where('session_id', $sessionId);
                }
                if ($ipAddress) {
                    $q->orWhere('ip_address', $ipAddress);
                }
            });
        }

        return $query->exists();
    }

    /**
     * Increment specific interaction counter
     */
    protected function incrementInteractionCounter(int $listingId, string $type): void
    {
        $columnMap = [
            ListingInteraction::TYPE_PHONE_REVEAL => 'phone_reveals_count',
            ListingInteraction::TYPE_WHATSAPP_CLICK => 'whatsapp_clicks_count',
            ListingInteraction::TYPE_SHARE => 'shares_count',
        ];

        if (isset($columnMap[$type])) {
            Listing::where('id', $listingId)->increment($columnMap[$type]);
        }
    }

    /**
     * Get comprehensive analytics for a listing
     */
    public function getAnalytics(Listing $listing, ?string $period = '30days'): array
    {
        $startDate = $this->getStartDate($period);

        return [
            'overview' => [
                'impressions' => $listing->impressions_count,
                'clicks' => $listing->clicks_count,
                'views' => $listing->views_count,
                'unique_views' => $listing->unique_views_count,
                'ctr' => $this->calculateCTR($listing->impressions_count, $listing->clicks_count),
                'conversion_rate' => $this->calculateConversionRate($listing->views_count, $listing->phone_reveals_count + $listing->whatsapp_clicks_count),
            ],
            'engagement' => [
                'phone_reveals' => $listing->phone_reveals_count,
                'whatsapp_clicks' => $listing->whatsapp_clicks_count,
                'shares' => $listing->shares_count,
            ],
            'period_stats' => [
                'period' => $period,
                'impressions' => $this->getCountForPeriod(ListingImpression::class, $listing->id, $startDate),
                'clicks' => $this->getCountForPeriod(ListingClick::class, $listing->id, $startDate),
                'views' => $this->getCountForPeriod(\App\Models\ListingView::class, $listing->id, $startDate),
            ],
            'sources' => [
                'impressions_by_source' => $this->getSourceBreakdown(ListingImpression::class, $listing->id, $startDate),
                'clicks_by_source' => $this->getSourceBreakdown(ListingClick::class, $listing->id, $startDate),
            ],
            'trends' => [
                'daily_impressions' => $this->getDailyTrend(ListingImpression::class, $listing->id, $startDate),
                'daily_clicks' => $this->getDailyTrend(ListingClick::class, $listing->id, $startDate),
            ],
        ];
    }

    /**
     * Get analytics for entire business/category (for advertisers)
     */
    public function getPlatformAnalytics(?string $category = null, ?string $period = '30days'): array
    {
        $startDate = $this->getStartDate($period);
        
        $query = Listing::query()->publicVisible();
        
        if ($category) {
            $query->where('category', $category);
        }

        $listings = $query->get();

        return [
            'total_listings' => $listings->count(),
            'total_impressions' => $listings->sum('impressions_count'),
            'total_clicks' => $listings->sum('clicks_count'),
            'total_views' => $listings->sum('views_count'),
            'average_ctr' => $this->calculateCTR(
                $listings->sum('impressions_count'),
                $listings->sum('clicks_count')
            ),
            'top_performing_listings' => $this->getTopPerformingListings($category, 10),
            'category_breakdown' => $this->getCategoryBreakdown($period),
        ];
    }

    /**
     * Calculate Click-Through Rate
     */
    protected function calculateCTR(int $impressions, int $clicks): float
    {
        if ($impressions === 0) {
            return 0.0;
        }
        return round(($clicks / $impressions) * 100, 2);
    }

    /**
     * Calculate Conversion Rate (views to contacts)
     */
    protected function calculateConversionRate(int $views, int $conversions): float
    {
        if ($views === 0) {
            return 0.0;
        }
        return round(($conversions / $views) * 100, 2);
    }

    /**
     * Get start date based on period
     */
    protected function getStartDate(string $period): \DateTime
    {
        return match($period) {
            '7days' => now()->subDays(7),
            '30days' => now()->subDays(30),
            '90days' => now()->subDays(90),
            'year' => now()->subYear(),
            default => now()->subDays(30),
        };
    }

    /**
     * Get count for specific period
     */
    protected function getCountForPeriod(string $model, int $listingId, \DateTime $startDate): int
    {
        $dateColumn = match($model) {
            ListingImpression::class => 'impressed_at',
            ListingClick::class => 'clicked_at',
            \App\Models\ListingView::class => 'viewed_at',
            default => 'created_at',
        };

        return $model::where('listing_id', $listingId)
            ->where($dateColumn, '>=', $startDate)
            ->count();
    }

    /**
     * Get source breakdown
     */
    protected function getSourceBreakdown(string $model, int $listingId, \DateTime $startDate): array
    {
        $dateColumn = $model === ListingImpression::class ? 'impressed_at' : 'clicked_at';

        return $model::where('listing_id', $listingId)
            ->where($dateColumn, '>=', $startDate)
            ->select('source', DB::raw('count(*) as count'))
            ->groupBy('source')
            ->pluck('count', 'source')
            ->toArray();
    }

    /**
     * Get daily trend data
     */
    protected function getDailyTrend(string $model, int $listingId, \DateTime $startDate): array
    {
        $dateColumn = $model === ListingImpression::class ? 'impressed_at' : 'clicked_at';

        return $model::where('listing_id', $listingId)
            ->where($dateColumn, '>=', $startDate)
            ->select(DB::raw('DATE(' . $dateColumn . ') as date'), DB::raw('count(*) as count'))
            ->groupBy('date')
            ->orderBy('date')
            ->pluck('count', 'date')
            ->toArray();
    }

    /**
     * Get top performing listings
     */
    protected function getTopPerformingListings(?string $category, int $limit = 10): array
    {
        $query = Listing::query()
            ->publicVisible()
            ->select('id', 'title', 'slug', 'impressions_count', 'clicks_count', 'views_count')
            ->orderByDesc('clicks_count');

        if ($category) {
            $query->where('category', $category);
        }

        return $query->limit($limit)
            ->get()
            ->map(function ($listing) {
                return [
                    'id' => $listing->id,
                    'title' => $listing->title,
                    'slug' => $listing->slug,
                    'impressions' => $listing->impressions_count,
                    'clicks' => $listing->clicks_count,
                    'views' => $listing->views_count,
                    'ctr' => $this->calculateCTR($listing->impressions_count, $listing->clicks_count),
                ];
            })
            ->toArray();
    }

    /**
     * Get category breakdown
     */
    protected function getCategoryBreakdown(string $period): array
    {
        return Listing::query()
            ->publicVisible()
            ->select('category', 
                DB::raw('SUM(impressions_count) as total_impressions'),
                DB::raw('SUM(clicks_count) as total_clicks'),
                DB::raw('COUNT(*) as listing_count')
            )
            ->groupBy('category')
            ->get()
            ->map(function ($item) {
                return [
                    'category' => $item->category,
                    'listings' => $item->listing_count,
                    'impressions' => $item->total_impressions,
                    'clicks' => $item->total_clicks,
                    'ctr' => $this->calculateCTR($item->total_impressions, $item->total_clicks),
                ];
            })
            ->toArray();
    }
}
