<?php

namespace App\Services;

use App\Models\InventoryBatch;
use App\Models\InventoryLog;
use App\Models\Listing;
use Illuminate\Support\Facades\DB;

class InventoryService
{
    /**
     * Add new stock (Restock) to Store (Warehouse) or Shop (POS)
     */
    public function addStock(
        Listing $listing,
        int $qty,
        float $costPrice,
        ?string $supplier = null,
        ?string $referenceNote = null,
        ?int $userId = null,
        string $targetLocation = 'store' // 'store' or 'shop'
    ): InventoryBatch {
        return DB::transaction(function () use ($listing, $qty, $costPrice, $supplier, $referenceNote, $userId, $targetLocation) {
            $isStore = $targetLocation === 'store';

            // Create Batch
            $batch = InventoryBatch::create([
                'business_id' => $listing->business_id,
                'listing_id' => $listing->id,
                'original_quantity' => $qty,
                'remaining_store_quantity' => $isStore ? $qty : 0,
                'remaining_shop_quantity' => $isStore ? 0 : $qty,
                'cost_price' => $costPrice,
                'received_at' => now(),
                'supplier' => $supplier,
                'reference_note' => $referenceNote,
            ]);

            // Update Listing Quantities
            if ($isStore) {
                $listing->store_quantity = ($listing->store_quantity ?? 0) + $qty;
            } else {
                $listing->quantity = ($listing->quantity ?? 0) + $qty;
            }
            $listing->save();

            // Log
            InventoryLog::create([
                'business_id' => $listing->business_id,
                'listing_id' => $listing->id,
                'batch_id' => $batch->id,
                'change_amount' => $qty,
                'new_quantity' => $isStore ? $listing->store_quantity : $listing->quantity,
                'reason' => 'restock',
                'location' => $targetLocation,
                'reference_id' => "BATCH-{$batch->id}",
                'performed_by_user_id' => $userId,
            ]);

            return $batch;
        });
    }

    /**
     * Move stock from Store to Shop
     */
    public function moveStockToShop(
        Listing $listing,
        int $qty,
        ?int $userId = null
    ): void {
        DB::transaction(function () use ($listing, $qty, $userId) {
            $remainingToMove = $qty;

            // Get batches with store stock, Oldest First (FIFO)
            $batches = InventoryBatch::where('listing_id', $listing->id)
                ->where('remaining_store_quantity', '>', 0)
                ->orderBy('received_at', 'asc')
                ->lockForUpdate()
                ->get();

            foreach ($batches as $batch) {
                if ($remainingToMove <= 0) break;

                $take = min($batch->remaining_store_quantity, $remainingToMove);

                $batch->remaining_store_quantity -= $take;
                $batch->remaining_shop_quantity += $take;
                $batch->save();

                $remainingToMove -= $take;
            }

            if ($remainingToMove > 0) {
                throw new \Exception("Insufficient store stock to move $qty items.");
            }

            // Update Listing Aggregates
            $listing->store_quantity = ($listing->store_quantity ?? 0) - $qty;
            $listing->quantity = ($listing->quantity ?? 0) + $qty;
            $listing->save();

            // Log Store Deduction (Transfer Out)
            InventoryLog::create([
                'business_id' => $listing->business_id,
                'listing_id' => $listing->id,
                'change_amount' => -$qty,
                'new_quantity' => $listing->store_quantity,
                'reason' => 'transfer_out',
                'location' => 'store',
                'performed_by_user_id' => $userId,
            ]);

            // Log Shop Addition (Transfer In)
            InventoryLog::create([
                'business_id' => $listing->business_id,
                'listing_id' => $listing->id,
                'change_amount' => $qty,
                'new_quantity' => $listing->quantity,
                'reason' => 'transfer_in',
                'location' => 'shop',
                'performed_by_user_id' => $userId,
            ]);
        });
    }

    /**
     * Deduct stock from Shop (Sale) using FIFO
     */
    public function deductShopStock(
        Listing $listing,
        int $qty,
        string $reason = 'sale',
        ?string $referenceId = null,
        ?int $userId = null
    ): void {
        $this->removeStock($listing, $qty, 'shop', $reason, $referenceId, $userId);
    }

    /**
     * Remove stock from Store or Shop (Damage, Shrinkage, Correction)
     */
    public function removeStock(
        Listing $listing,
        int $qty,
        string $location = 'shop', // 'store' or 'shop'
        string $reason = 'manual_adjustment',
        ?string $referenceId = null,
        ?int $userId = null
    ): void {
        DB::transaction(function () use ($listing, $qty, $location, $reason, $referenceId, $userId) {
            $isStore = $location === 'store';
            $currentTotal = $isStore ? ($listing->store_quantity ?? 0) : ($listing->quantity ?? 0);

            if ($qty > $currentTotal) {
                $locationName = $isStore ? 'store' : 'shop';
                throw new \Exception("Insufficient stock in $locationName. You cannot remove more than available ($currentTotal).");
            }

            $remainingToDeduct = $qty;
            $batchColumn = $isStore ? 'remaining_store_quantity' : 'remaining_shop_quantity';

            // Get batches with stock in target location, Oldest First (FIFO)
            $batches = InventoryBatch::where('listing_id', $listing->id)
                ->where($batchColumn, '>', 0)
                ->orderBy('received_at', 'asc')
                ->lockForUpdate()
                ->get();

            $currentListingQty = $currentTotal;

            foreach ($batches as $batch) {
                if ($remainingToDeduct <= 0) break;

                $take = min($batch->$batchColumn, $remainingToDeduct);

                $batch->$batchColumn -= $take;
                $batch->save();

                $currentListingQty -= $take;

                // Log per batch usage for FIFO Profit tracking
                InventoryLog::create([
                    'business_id' => $listing->business_id,
                    'listing_id' => $listing->id,
                    'batch_id' => $batch->id,
                    'change_amount' => -$take,
                    'new_quantity' => $currentListingQty,
                    'reason' => $reason,
                    'location' => $location,
                    'reference_id' => $referenceId,
                    'performed_by_user_id' => $userId,
                ]);

                $remainingToDeduct -= $take;
            }

            // If there's still quantity to deduct (Legacy stock not in batches)
            if ($remainingToDeduct > 0) {
                $currentListingQty -= $remainingToDeduct;
                InventoryLog::create([
                    'business_id' => $listing->business_id,
                    'listing_id' => $listing->id,
                    'batch_id' => null,
                    'change_amount' => -$remainingToDeduct,
                    'new_quantity' => $currentListingQty,
                    'reason' => $reason . ' (legacy_stock)',
                    'location' => $location,
                    'reference_id' => $referenceId,
                    'performed_by_user_id' => $userId,
                ]);
            }

            // Update Listing Aggregate
            if ($isStore) {
                $listing->store_quantity = $currentListingQty;
            } else {
                $listing->quantity = $currentListingQty;
            }
            $listing->save();
        });
    }
}
