<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;

class ImageRecognitionService
{
    /**
     * Detect labels/tags in an image using Google Cloud Vision API
     * 
     * @param string $imagePath Path to image in storage
     * @return array Array of detected labels
     */
    public function detectLabels(string $imagePath): array
    {
        $apiKey = config('services.google_vision.api_key');
        
        // If no API key configured, return empty array
        if (!$apiKey) {
            Log::warning('Google Vision API key not configured. Skipping image tagging.');
            return [];
        }

        try {
            // Read image file and encode to base64
            $fullPath = Storage::disk('public')->path($imagePath);
            
            if (!file_exists($fullPath)) {
                return [];
            }

            $imageContent = base64_encode(file_get_contents($fullPath));

            // Call Google Vision API
            $response = Http::timeout(10)->post(
                "https://vision.googleapis.com/v1/images:annotate?key={$apiKey}",
                [
                    'requests' => [
                        [
                            'image' => [
                                'content' => $imageContent,
                            ],
                            'features' => [
                                [
                                    'type' => 'LABEL_DETECTION',
                                    'maxResults' => 10,
                                ],
                            ],
                        ],
                    ],
                ]
            );

            if (!$response->successful()) {
                Log::error('Google Vision API error', [
                    'status' => $response->status(),
                    'body' => $response->body(),
                ]);
                return [];
            }

            $data = $response->json();
            
            // Log the full response for debugging
            Log::info('Google Vision API response', [
                'has_responses' => isset($data['responses']),
                'has_labels' => isset($data['responses'][0]['labelAnnotations']),
                'response_data' => $data,
            ]);
            
            if (!isset($data['responses'][0]['labelAnnotations'])) {
                Log::warning('No labelAnnotations in response', ['response' => $data]);
                return [];
            }

            // Extract labels with confidence > 0.7
            $labels = [];
            foreach ($data['responses'][0]['labelAnnotations'] as $annotation) {
                if (isset($annotation['score']) && $annotation['score'] >= 0.7) {
                    $labels[] = [
                        'label' => $annotation['description'] ?? '',
                        'confidence' => round($annotation['score'], 2),
                    ];
                }
            }

            return $labels;

        } catch (\Exception $e) {
            Log::error('Image recognition failed', [
                'error' => $e->getMessage(),
                'path' => $imagePath,
            ]);
            return [];
        }
    }

    /**
     * Fallback: Basic image analysis without external API
     * Analyzes image properties like dominant colors, dimensions, etc.
     * 
     * @param string $imagePath Path to image in storage
     * @return array Array of basic tags
     */
    public function analyzeImageBasic(string $imagePath): array
    {
        try {
            $fullPath = Storage::disk('public')->path($imagePath);
            
            if (!file_exists($fullPath)) {
                return [];
            }

            $imageInfo = getimagesize($fullPath);
            if (!$imageInfo) {
                return [];
            }

            $tags = [];
            
            // Orientation
            $width = $imageInfo[0];
            $height = $imageInfo[1];
            
            if ($width > $height * 1.3) {
                $tags[] = ['label' => 'landscape', 'confidence' => 1.0];
            } elseif ($height > $width * 1.3) {
                $tags[] = ['label' => 'portrait', 'confidence' => 1.0];
            } else {
                $tags[] = ['label' => 'square', 'confidence' => 1.0];
            }

            // Image size category
            $megapixels = ($width * $height) / 1000000;
            if ($megapixels > 5) {
                $tags[] = ['label' => 'high-resolution', 'confidence' => 1.0];
            }

            return $tags;

        } catch (\Exception $e) {
            Log::error('Basic image analysis failed', [
                'error' => $e->getMessage(),
                'path' => $imagePath,
            ]);
            return [];
        }
    }

    /**
     * Detect web entities and similar images for reverse image search
     * Uses Google Vision API's WEB_DETECTION feature
     * 
     * @param string $imagePath Path to image in storage
     * @return array Array with web entities and visual matches
     */
    public function detectWebEntities(string $imagePath): array
    {
        $apiKey = config('services.google_vision.api_key');
        
        if (!$apiKey) {
            return ['entities' => [], 'similar_images' => []];
        }

        try {
            $fullPath = Storage::disk('public')->path($imagePath);
            
            if (!file_exists($fullPath)) {
                return ['entities' => [], 'similar_images' => []];
            }

            $imageContent = base64_encode(file_get_contents($fullPath));

            // Call Google Vision API with WEB_DETECTION
            $response = Http::timeout(10)->post(
                "https://vision.googleapis.com/v1/images:annotate?key={$apiKey}",
                [
                    'requests' => [
                        [
                            'image' => [
                                'content' => $imageContent,
                            ],
                            'features' => [
                                [
                                    'type' => 'WEB_DETECTION',
                                    'maxResults' => 20,
                                ],
                                [
                                    'type' => 'LABEL_DETECTION',
                                    'maxResults' => 10,
                                ],
                            ],
                        ],
                    ],
                ]
            );

            if (!$response->successful()) {
                Log::error('Google Vision API WEB_DETECTION error', [
                    'status' => $response->status(),
                    'body' => $response->body(),
                ]);
                return ['entities' => [], 'similar_images' => []];
            }

            $data = $response->json();
            
            $entities = [];
            $labels = [];
            
            // Extract web entities (what the image represents)
            if (isset($data['responses'][0]['webDetection']['webEntities'])) {
                foreach ($data['responses'][0]['webDetection']['webEntities'] as $entity) {
                    if (isset($entity['score']) && $entity['score'] >= 0.5) {
                        $entities[] = [
                            'description' => $entity['description'] ?? '',
                            'score' => round($entity['score'], 2),
                        ];
                    }
                }
            }
            
            // Extract labels as fallback
            if (isset($data['responses'][0]['labelAnnotations'])) {
                foreach ($data['responses'][0]['labelAnnotations'] as $annotation) {
                    if (isset($annotation['score']) && $annotation['score'] >= 0.7) {
                        $labels[] = $annotation['description'] ?? '';
                    }
                }
            }

            return [
                'entities' => $entities,
                'labels' => $labels,
            ];

        } catch (\Exception $e) {
            Log::error('Web entity detection failed', [
                'error' => $e->getMessage(),
                'path' => $imagePath,
            ]);
            return ['entities' => [], 'labels' => []];
        }
    }
}
