<?php
namespace App\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class GeminiService
{
    protected string $apiKey;
    protected string $baseUrl = 'https://generativelanguage.googleapis.com/v1/models/gemini-2.5-flash:generateContent';

    public function __construct()
    {
        $this->apiKey = config('services.gemini.api_key', env('GEMINI_API_KEY'));
    }

    /**
     * Generate a summary for a POS order using Gemini.
     *
     * @param array $orderData
     * @param array $itemCosts Map of listing_id => total_cost
     * @return string
     */
    public function generateOrderSummary(array $orderData, array $itemCosts = []): string
    {
        if (empty($this->apiKey)) {
            Log::warning('Gemini API key not configured.');
            return "AI summary data is currently unavailable. Please check system configuration.";
        }

        try {
            $prompt = $this->buildPrompt($orderData, $itemCosts);
            
            $response = Http::withHeaders([
                'Content-Type' => 'application/json',
            ])->post("{$this->baseUrl}?key={$this->apiKey}", [
                'contents' => [
                    [
                        'parts' => [
                            ['text' => $prompt]
                        ]
                    ]
                ],
                'generationConfig' => [
                    'temperature' => 0.7,
                    'maxOutputTokens' => 2048,
                    'topP' => 0.95,
                    'topK' => 40,
                ],
                'safetySettings' => [
                    [
                        'category' => 'HARM_CATEGORY_HARASSMENT',
                        'threshold' => 'BLOCK_ONLY_HIGH'
                    ],
                    [
                        'category' => 'HARM_CATEGORY_HATE_SPEECH',
                        'threshold' => 'BLOCK_ONLY_HIGH'
                    ],
                    [
                        'category' => 'HARM_CATEGORY_SEXUALLY_EXPLICIT',
                        'threshold' => 'BLOCK_ONLY_HIGH'
                    ],
                    [
                        'category' => 'HARM_CATEGORY_DANGEROUS_CONTENT',
                        'threshold' => 'BLOCK_ONLY_HIGH'
                    ]
                ]
            ]);

            if (!$response->successful()) {
                Log::error('Gemini API Error', [
                    'status' => $response->status(),
                    'body' => $response->body(),
                ]);
                return "Unable to generate summary at this time.";
            }

            $data = $response->json();
            
            // Check if response was blocked
            if (isset($data['promptFeedback']['blockReason'])) {
                Log::warning('Gemini response blocked', [
                    'reason' => $data['promptFeedback']['blockReason']
                ]);
                return "Unable to generate summary due to content restrictions.";
            }
            
            // Extract text from Gemini response structure
            $text = $data['candidates'][0]['content']['parts'][0]['text'] ?? null;
            
            // Check finish reason
            $finishReason = $data['candidates'][0]['finishReason'] ?? null;
            if ($finishReason === 'SAFETY') {
                Log::warning('Gemini response stopped due to safety', $data);
                return "Summary generation was restricted due to safety filters.";
            }
            
            if (!$text) {
                Log::warning('No text in Gemini response', $data);
                return "No summary generated.";
            }

            return trim($text);
            
        } catch (\Exception $e) {
            Log::error('Gemini Service Exception', ['message' => $e->getMessage()]);
            return "An error occurred while generating the summary.";
        }
    }

    /**
     * Generate AI insights for inventory health.
     *
     * @param array $summaryData Quantitative metrics (velocity, capital, health_score)
     * @param array $anomalies List of specific issues (dead stock, overstock, stockouts)
     * @return string
     */
    public function generateInventoryInsights(array $summaryData, array $anomalies): string
    {
        if (empty($this->apiKey)) {
            return "AI analysis is currently unavailable. Please check system configuration.";
        }

        try {
            $anomaliesList = "";
            foreach ($anomalies as $type => $items) {
                if (empty($items)) continue;
                $count = count($items);
                $examples = collect($items)->take(3)->pluck('title')->implode(', ');
                $anomaliesList .= "- {$type} ({$count} items): e.g., {$examples}\n";
            }

            $prompt = "You are a Retail Growth AI for Yilick. Analyze this inventory health data and provide strategic advice.

            SUMMARY METRICS:
            - Health Score: {$summaryData['health_score']}/100
            - Capital Tied Up: {$summaryData['total_capital_tied_up']} {$summaryData['currency']}
            - Monthly Run Rate: {$summaryData['total_monthly_velocity']} units
            
            IDENTIFIED ISSUES:
            {$anomaliesList}
            
            YOUR TASK:
            1. **The Verdict**: 1 sentence on overall stock health.
            2. **Deep Dive**: Why is capital tied up? (Analyze dead/overstock).
            3. **Growth Lever**: 1-2 specific actions to improve cash flow (e.g., clearance sales for dead stock, bundle deals).
            
            FORMAT:
            Keep it professional, concise, and ACTIONABLE. Use plain text only. No markdown.
            
            === STRATEGIC ADVISE ===
            [Your analysis here]";

            $response = Http::post("{$this->baseUrl}?key={$this->apiKey}", [
                'contents' => [['parts' => [['text' => $prompt]]]]
            ]);

            if (!$response->successful()) {
                Log::error('Gemini Inventory Insights API Failure', [
                    'status' => $response->status(),
                    'body' => $response->body(),
                    'prompt_length' => strlen($prompt)
                ]);
                return "Strategic analysis temporarily unavailable.";
            }

            $data = $response->json();
            
            if (!isset($data['candidates'][0]['content']['parts'][0]['text'])) {
                Log::warning('Gemini response missing text', ['data' => $data]);
                return "No insights generated.";
            }

            return trim($data['candidates'][0]['content']['parts'][0]['text']);
        } catch (\Exception $e) {
            Log::error('Gemini Inventory Health Error', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return "Unable to perform strategic analysis at this time.";
        }
    }

    protected function buildPrompt(array $order, array $itemCosts): string
    {
        $itemsList = "";
        $items = $order['items'] ?? [];
        
        if (!is_array($items)) {
            $items = [];
        }

        $totalCost = 0;

        foreach ($items as $item) {
            $qty = $item['quantity'] ?? $item['qty'] ?? 1;
            $title = $item['title'] ?? 'Unknown Item';
            $price = $item['price'] ?? 0;
            $totalItemPrice = $item['total'] ?? ($price * $qty);
            $itemId = $item['id'] ?? null;
            
            // Get detailed cost info
            $costData = $itemCosts[$itemId] ?? null;
            $batchesText = "";
            
            if ($costData && is_array($costData)) {
                $cost = $costData['total_cost'] ?? 0;
                // Build explicit calculation text
                foreach (($costData['batches'] ?? []) as $batch) {
                    $bQty = $batch['quantity'];
                    $bUnit = $batch['unit_cost'];
                    $bTotal = $batch['total_chunk_cost'];
                    $batchesText .= "       - Batch Source: {$bQty} units @ {$bUnit} purchase price = {$bTotal} cost\n";
                }
            } else {
                // Fallback loop if no batch data found
                $unitCost = $item['cost_price'] ?? 0;
                $cost = $unitCost * $qty;
                $batchesText = "       - Estimated: {$qty} units @ {$unitCost} purchase price (fallback) = {$cost} cost\n";
            }
            
            $totalCost += $cost;
            $profit = $totalItemPrice - $cost;
            $margin = $totalItemPrice > 0 ? round(($profit / $totalItemPrice) * 100, 1) : 0;
            
            $itemsList .= "Item: {$title} (x{$qty})\n";
            $itemsList .= "   Revenue: {$totalItemPrice} (Sold @ {$price}/unit)\n";
            $itemsList .= "   Cost Breakdown:\n{$batchesText}";
            $itemsList .= "   -> Total COGS: {$cost} | Total Profit: {$profit} | Margin: {$margin}%\n\n";
        }

        $totalRevenue = $order['total_amount'] ?? 0;
        $totalProfit = $totalRevenue - $totalCost;
        $totalMargin = $totalRevenue > 0 ? round(($totalProfit / $totalRevenue) * 100, 1) : 0;
        
        $customer = data_get($order, 'customer.name', 'Walk-in Customer');
        $date = $order['created_at'] ?? now()->toDateTimeString();

        return "You are an Expert Retail Strategist for the Yilick App. Provide a concise, cost-effective financial analysis.
        
        YOUR MANDATE:
        1. **Financials**: Quick profit/margin summary.
        2. **Pricing**: Is it sustainable?
        3. **Costs**: Highlight rising costs (only if critical).
        4. **Strategy**: 1-2 specific actions.
        
        OUTPUT FORMAT (Plain text. Keep it brief to save tokens):
        
        FINANCIALS
        [1 sentence summary]
        - Net Profit: [Amount]
        - Margin: {$totalMargin}%
        
        STRATEGY & PRICING
        [2 sentences max. Critique pricing and costs together.]
        
        ACTION PLAN
        - [Item]: [Action]. [Brief reason].
        
        STRICT RULES:
        - Be concise. No fluff.
        - Analyze, don't just describe.
        - NO markdown symbols.
        
        === FINANCIAL DATA ===
        Customer: $customer
        Date: $date
        
        GRAND TOTALS:
        Total Revenue: $totalRevenue
        Total Cost (COGS): $totalCost
        NET PROFIT: $totalProfit
        OVERALL MARGIN: $totalMargin%
        
        === ITEM BREAKDOWN ===
        $itemsList";
    }
}