<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class AIListingGeneratorService
{
    protected string $apiKey;
    protected string $model;

    public function __construct()
    {
        $this->apiKey = config('services.openai.api_key');
        $this->model = config('services.openai.model', 'gpt-4o-mini');
    }

    /**
     * Generate improved title, description, and tags for a product listing
     * 
     * @param string $shortTitle Short product title from seller
     * @param string|null $category Optional category for context
     * @param string|null $additionalContext Optional additional context
     * @return array ['title' => string, 'description' => string, 'tags' => array]
     */
    public function generateListingContent(
        string $shortTitle,
        ?string $category = null,
        ?string $additionalContext = null
    ): array {
        if (!$this->apiKey) {
            Log::warning('OpenAI API key not configured. Cannot generate listing content.');
            return $this->getFallbackContent($shortTitle);
        }

        try {
            $prompt = $this->buildPrompt($shortTitle, $category, $additionalContext);
            
            $response = Http::timeout(30)
                ->withHeaders([
                    'Authorization' => 'Bearer ' . $this->apiKey,
                    'Content-Type' => 'application/json',
                ])
                ->post('https://api.openai.com/v1/chat/completions', [
                    'model' => $this->model,
                    'messages' => [
                        [
                            'role' => 'system',
                            'content' => 'You are an expert e-commerce copywriter specializing in creating compelling product listings for classified ads and marketplaces. Generate SEO-optimized, engaging content that attracts buyers.'
                        ],
                        [
                            'role' => 'user',
                            'content' => $prompt
                        ]
                    ],
                    'temperature' => 0.7,
                    'max_tokens' => 800,
                    'response_format' => ['type' => 'json_object'],
                ]);

            if (!$response->successful()) {
                Log::error('OpenAI API error', [
                    'status' => $response->status(),
                    'body' => $response->body(),
                ]);
                return $this->getFallbackContent($shortTitle);
            }

            $data = $response->json();
            $content = $data['choices'][0]['message']['content'] ?? null;

            if (!$content) {
                return $this->getFallbackContent($shortTitle);
            }

            $generated = json_decode($content, true);

            return [
                'title' => $generated['title'] ?? $shortTitle,
                'description' => $generated['description'] ?? '',
                'tags' => $generated['tags'] ?? [],
                'meta_title' => $generated['meta_title'] ?? null,
                'meta_description' => $generated['meta_description'] ?? null,
            ];

        } catch (\Exception $e) {
            Log::error('AI listing generation failed', [
                'error' => $e->getMessage(),
                'title' => $shortTitle,
            ]);
            return $this->getFallbackContent($shortTitle);
        }
    }

    /**
     * Build the prompt for AI generation
     */
    protected function buildPrompt(string $shortTitle, ?string $category, ?string $additionalContext): string
    {
        $prompt = "Generate an improved product listing for a classified ads marketplace.\n\n";
        $prompt .= "Product: {$shortTitle}\n";
        
        if ($category) {
            $prompt .= "Category: {$category}\n";
        }
        
        if ($additionalContext) {
            $prompt .= "Additional Context: {$additionalContext}\n";
        }

        $prompt .= "\nGenerate the following in JSON format:\n";
        $prompt .= "1. **title**: An improved, attention-grabbing title (max 100 characters). Make it descriptive and include key features.\n";
        $prompt .= "2. **description**: A compelling product description (200-400 words) that:\n";
        $prompt .= "   - Highlights key features and benefits\n";
        $prompt .= "   - Uses bullet points for easy reading\n";
        $prompt .= "   - Includes a call-to-action\n";
        $prompt .= "   - Is SEO-friendly\n";
        $prompt .= "3. **tags**: An array of 5-10 relevant search tags/keywords\n";
        $prompt .= "4. **meta_title**: SEO meta title (max 60 characters)\n";
        $prompt .= "5. **meta_description**: SEO meta description (max 160 characters)\n\n";
        $prompt .= "Return ONLY valid JSON with these exact keys: title, description, tags, meta_title, meta_description";

        return $prompt;
    }

    /**
     * Fallback content when AI is unavailable
     */
    protected function getFallbackContent(string $shortTitle): array
    {
        return [
            'title' => $shortTitle,
            'description' => "High-quality {$shortTitle} available for sale. Contact seller for more details.",
            'tags' => $this->extractSimpleTags($shortTitle),
            'meta_title' => $shortTitle,
            'meta_description' => "Buy {$shortTitle} - Quality product at great price",
        ];
    }

    /**
     * Extract simple tags from title as fallback
     */
    protected function extractSimpleTags(string $title): array
    {
        // Remove common words and split
        $commonWords = ['the', 'a', 'an', 'and', 'or', 'but', 'in', 'on', 'at', 'to', 'for', 'of', 'with'];
        $words = preg_split('/[\s,\-_]+/', strtolower($title));
        
        $tags = array_filter($words, function($word) use ($commonWords) {
            return strlen($word) > 2 && !in_array($word, $commonWords);
        });

        return array_values(array_slice($tags, 0, 5));
    }
}
