<?php

namespace App\Notifications;

use App\Models\Listing;
use App\Services\FcmService;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class AdRejectedNotification extends Notification implements ShouldQueue
{
    use Queueable;

    public function __construct(public Listing $listing)
    {
    }

    public function via($notifiable): array
    {
        return ['mail', 'database'];
    }

    public function toMail($notifiable): MailMessage
    {
        $url = config('app.url') . "/my-ads?tab=declined"; 

        $mailMessage = (new MailMessage)
            ->subject('Listing Update Required')
            ->greeting('Hello ' . $notifiable->full_name . ',')
            ->line('Thank you for choosing ' . config('app.name') . ' to list your items.')
            ->line('Our moderation team (or automated system) has reviewed your ad "' . $this->listing->title . '", and we\'ve determined that it requires some adjustments before it can be published.');

        if ($this->listing->decline_reason) {
            $reasonDetail = Listing::getRejectionReasonDetail($this->listing->decline_reason);
            $mailMessage->line('**Reason for Moderation Action:**')
                ->line('> ' . $reasonDetail);
        }

        return $mailMessage
            ->line('We maintain these standards to ensure a safe, reliable, and high-quality experience for all our users. Most rejections are easily fixed by updating the description, price, or images.')
            ->line('**What should you do next?**')
            ->line('1. Click the button below to view your listing.')
            ->line('2. Update the fields mentioned in the reason above.')
            ->line('3. Re-submit your ad for review.')
            ->action('Fix and Re-submit My Ad', $url)
            ->line('If you believe this was an error, please don\'t hesitate to contact our support team for a manual second-look.')
            ->salutation('Best regards,  
The ' . config('app.name') . ' Quality Assurance Team');
    }

    public function toArray($notifiable): array
    {
        $title = 'Ad Declined';
        $body = 'Your ad "' . $this->listing->title . '" was not published.';
        $data = [
            'type' => 'listing_rejected',
            'listing_id' => $this->listing->id,
            'reason' => $this->listing->decline_reason,
        ];

        // Trigger FCM
        if ($notifiable->id) {
            try {
                app(FcmService::class)->sendToUser(
                    $notifiable->id,
                    $title,
                    $body,
                    $data
                );
            } catch (\Throwable $e) {
                \Log::warning('Failed to send FCM notification (AdRejected)', [
                    'user_id' => $notifiable->id,
                    'error' => $e->getMessage(),
                ]);
            }
        }

        return [
            'title' => $title,
            'body' => $body,
            'data' => $data,
        ];
    }
}
