<?php

namespace App\Notifications;

use App\Models\Listing;
use App\Services\FcmService;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class AdPublishedNotification extends Notification implements ShouldQueue
{
    use Queueable;

    public function __construct(public Listing $listing)
    {
    }

    public function via($notifiable): array
    {
        return ['mail', 'database'];
    }

    public function toMail($notifiable): MailMessage
    {
        $url = config('app.url') . "/listings/{$this->listing->slug}";

        return (new MailMessage)
            ->subject('Ad Published')
            ->greeting('Hello ' . $notifiable->full_name . ',')
            ->line('Great news! Your ad "' . $this->listing->title . '" has been reviewed and published.')
            ->line('It is now visible to thousands of potential customers on ' . config('app.name') . '.')
            ->action('View My Ad', $url)
            ->line('Thank you for using ' . config('app.name') . '!')
            ->salutation('Best regards,  
The ' . config('app.name') . ' Team');
    }

    public function toArray($notifiable): array
    {
        $title = 'Ad Published!';
        $body = 'Your ad "' . $this->listing->title . '" is now live.';
        $data = [
            'type' => 'listing_published',
            'listing_id' => $this->listing->id,
            'slug' => $this->listing->slug,
        ];

        // Trigger FCM
        if ($notifiable->id) {
            try {
                app(FcmService::class)->sendToUser(
                    $notifiable->id,
                    $title,
                    $body,
                    $data
                );
            } catch (\Throwable $e) {
                \Log::warning('Failed to send FCM notification (AdPublished)', [
                    'user_id' => $notifiable->id,
                    'error' => $e->getMessage(),
                ]);
            }
        }

        return [
            'title' => $title,
            'body' => $body,
            'data' => $data,
        ];
    }
}
