<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class UserSubscription extends Model
{
    protected $fillable = [
        'user_id',
        'business_id',
        'package_id',
        'status',
        'starts_at',
        'expires_at',
        'listings_used',
        'featured_listings_used',
        'bumps_used_this_month',
        'last_bump_reset',
        'amount_paid',
        'payment_method',
        'transaction_id',
        'cancelled_at',
        'notes',
    ];

    protected $casts = [
        'starts_at' => 'datetime',
        'expires_at' => 'datetime',
        'last_bump_reset' => 'datetime',
        'cancelled_at' => 'datetime',
        'amount_paid' => 'decimal:2',
        'featured_listings_used' => 'integer',
        'bumps_used_this_month' => 'integer',
        'auto_renew' => 'boolean',
    ];

    public function getListingsUsedAttribute(): int
    {
        if (!$this->business_id) {
            return 0;
        }

        return Listing::where('business_id', $this->business_id)
            ->whereIn('status', [Listing::STATUS_PUBLISHED, Listing::STATUS_REVIEW])
            ->count();
    }

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public function package(): BelongsTo
    {
        return $this->belongsTo(PremiumPackage::class, 'package_id');
    }

    public function business(): BelongsTo
    {
        return $this->belongsTo(Business::class);
    }

    public function listings(): HasMany
    {
        return $this->hasMany(Listing::class, 'subscription_id');
    }

    public function isActive(): bool
    {
        return $this->status === 'active' && $this->starts_at <= now() && $this->expires_at > now();
    }

    public function isExpired(): bool
    {
        return $this->expires_at <= now();
    }

    public function canCreateListing(): bool
    {
        return $this->isActive() && $this->listings_used < $this->package->max_listings;
    }

    public function canFeatureListing(): bool
    {
        return $this->isActive() && $this->featured_listings_used < $this->package->max_featured_listings;
    }

    public function canBumpListing(): bool
    {
        $this->resetBumpsIfNeeded();
        return $this->isActive() && $this->bumps_used_this_month < $this->package->max_bumps_per_month;
    }

    protected function resetBumpsIfNeeded(): void
    {
        if (!$this->last_bump_reset || $this->last_bump_reset->month !== now()->month) {
            $this->update([
                'bumps_used_this_month' => 0,
                'last_bump_reset' => now(),
            ]);
        }
    }
}
