<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Str;

class Listing extends Model
{
    use SoftDeletes;

    public const STATUS_DRAFT = 'draft';
    public const STATUS_PUBLISHED = 'published';
    public const STATUS_REVIEW = 'review';
    public const STATUS_REJECTED = 'rejected';
    public const STATUS_SOLD = 'sold';
    public const STATUS_EXPIRED = 'expired';

    public const REJECTION_REASONS = [
        'prohibited_item' => 'Prohibited/illegal item',
        'restricted_item' => 'Restricted item without required proof/authorization',
        'stolen_goods' => 'Suspected stolen goods',
        'counterfeit' => 'Counterfeit or fake brand product',
        'fraud_scam' => 'Fraud/scam indicators detected',
        'misleading_claims' => 'Misleading or false claims in description',
        'bait_pricing' => 'Bait pricing (fake price to attract clicks)',
        'price_missing' => 'Price missing',
        'location_invalid' => 'Location missing/invalid',
        'unclear_title' => 'Title is unclear, spammy, or misleading',
        'vague_description' => 'Description too vague / not enough details',
        'wrong_category' => 'Wrong category chosen',
        'duplicate_listing' => 'Duplicate listing (same item posted repeatedly)',
        'duplicate_images' => 'Duplicate images reused across unrelated ads',
        'low_quality_images' => 'Images are low quality / unreadable',
        'mismatched_images' => 'Images don’t match the item being sold',
        'prohibited_images' => 'Images contain prohibited content',
        'impersonation_watermarks' => 'Watermarks/logos used to impersonate a brand or another seller',
        'copyright_infringement' => 'Use of copyrighted photos you don’t own',
        'contact_in_forbidden_format' => 'Contact details posted in a forbidden format',
        'disallowed_links' => 'External links that redirect users off-platform (disallowed)',
        'advance_payment_request' => 'Requests for advance payment outside platform safety rules',
        'sensitive_info_request' => 'Asking for sensitive info (OTP, PIN, full card details, etc.)',
        'harassment_hate_speech' => 'Harassment, hate speech, or discriminatory language',
        'adult_content' => 'Adult sexual content/services',
        'violent_content' => 'Violent, graphic, or shocking content',
        'dangerous_activities' => 'Encouraging dangerous/illegal activities',
        'malware_phishing' => 'Malware/phishing attempts (suspicious links/files)',
        'policy_evasion' => 'Policy evasion (trying to bypass moderation with coded language)',
    ];

    public static function getRejectionReasonDetail(string $reasonKey): string
    {
        return self::REJECTION_REASONS[$reasonKey] ?? 'Your ad does not meet our community guidelines.';
    }

    protected $fillable = [
        'business_id',
        'created_by_user_id',
        'category',
        'sub_category',
        'sub_sub_category',
        'title',
        'slug',
        'description',
        'price_amount',
        'purchase_price_amount',
        'currency',
        'status',
        'decline_reason',
        'quantity',
        'store_quantity',
        'sku',
        'barcode',
        'allow_pos',
        'location_country_code',
        'location_city',
        'location_district',
        'latitude',
        'longitude',
        'attributes',
        'custom_fields',
        'tags',
        'meta_title',
        'meta_description',
        'meta_keywords',
        'canonical_url',
        'published_at',
        'bumped_at',
        'promoted_until',
        'promoted_rank',
        'expires_at',
        'views_count',
        'unique_views_count',
        'impressions_count',
        'clicks_count',
        'phone_reveals_count',
        'whatsapp_clicks_count',
        'shares_count',
        'stock_alert_level',
        'pos_quick_access',
        'is_featured',
        'featured_until',
        'hot_deals',
        'subscription_id',
    ];

    /**
     * Decrement stock and log the change.
     */
    public function decrementStock(int $qty = 1, string $reason = 'sale', ?string $referenceId = null, ?int $performerId = null): void
    {
        // Ensure quantity is not null before decrementing
        if ($this->quantity === null) {
            $this->quantity = 0;
            $this->save();
        }

        if ($qty > $this->quantity) {
            throw new \Exception("Insufficient stock. Available: {$this->quantity}, Requested: $qty");
        }

        $this->decrement('quantity', $qty);
        
        InventoryLog::create([
            'business_id' => $this->business_id,
            'listing_id' => $this->id,
            'change_amount' => -$qty,
            'new_quantity' => $this->fresh()->quantity,
            'reason' => $reason,
            'reference_id' => $referenceId,
            'performed_by_user_id' => $performerId,
        ]);
        
        // Trigger alert if low stock (Logic to be handled by listener or job)
    }

    /**
     * Increment stock and log the change.
     */
    public function incrementStock(int $qty = 1, string $reason = 'restock', ?string $referenceId = null, ?int $performerId = null): void
    {
        $this->increment('quantity', $qty);
        
        InventoryLog::create([
            'business_id' => $this->business_id,
            'listing_id' => $this->id,
            'change_amount' => $qty,
            'new_quantity' => $this->refresh()->quantity,
            'reason' => $reason,
            'reference_id' => $referenceId,
            'performed_by_user_id' => $performerId,
        ]);
    }

    protected function casts(): array
    {
        return [
            'business_id' => 'integer',
            'created_by_user_id' => 'integer',
            'price_amount' => 'decimal:2',
            'purchase_price_amount' => 'decimal:2',
            'quantity' => 'integer',
            'store_quantity' => 'integer',
            'allow_pos' => 'boolean',
            'latitude' => 'decimal:7',
            'longitude' => 'decimal:7',
            'attributes' => 'array',
            'custom_fields' => 'array',
            'tags' => 'array',
            'published_at' => 'datetime',
            'bumped_at' => 'datetime',
            'promoted_until' => 'datetime',
            'expires_at' => 'datetime',
            'views_count' => 'integer',
            'unique_views_count' => 'integer',
            'impressions_count' => 'integer',
            'clicks_count' => 'integer',
            'phone_reveals_count' => 'integer',
            'whatsapp_clicks_count' => 'integer',
            'shares_count' => 'integer',
            'promoted_rank' => 'integer',
            'stock_alert_level' => 'integer',
            'pos_quick_access' => 'boolean',
            'is_featured' => 'boolean',
            'featured_until' => 'datetime',
            'hot_deals' => 'boolean',
            'subscription_id' => 'integer',
        ];
    }

    protected static function booted(): void
    {
        static::creating(function (Listing $listing) {
            if (empty($listing->slug)) {
                $listing->slug = self::generateUniqueSlug($listing->title);
            }

            if (empty($listing->sku)) {
                $listing->sku = self::generateUniqueSku();
            }

            if (empty($listing->barcode)) {
                $listing->barcode = self::generateUniqueBarcode();
            }

            if (empty($listing->status)) {
                $listing->status = self::STATUS_DRAFT;
            }
        });

    }

    public function business(): BelongsTo
    {
        return $this->belongsTo(Business::class);
    }

    public function createdBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by_user_id');
    }

    public function inventoryLogs(): HasMany
    {
        return $this->hasMany(InventoryLog::class);
    }

    public function media(): HasMany
    {
        return $this->hasMany(ListingMedia::class);
    }

    public function views(): HasMany
    {
        return $this->hasMany(ListingView::class);
    }

    public function impressions(): HasMany
    {
        return $this->hasMany(ListingImpression::class);
    }

    public function clicks(): HasMany
    {
        return $this->hasMany(ListingClick::class);
    }

    public function interactions(): HasMany
    {
        return $this->hasMany(ListingInteraction::class);
    }

    public function favoritedBy(): BelongsToMany
    {
        return $this->belongsToMany(User::class, 'user_favorite_listings');
    }

    public function isFavoritedBy(?User $user): bool
    {
        if (!$user) {
            return false;
        }

        return $this->favoritedBy()->where('user_id', $user->id)->exists();
    }

    public function scopePublicVisible($query)
    {
        return $query->where('status', self::STATUS_PUBLISHED)
            ->where(function ($q) {
                $q->whereNull('expires_at')->orWhere('expires_at', '>', now());
            });
    }

    public function scopeOrderedForFeed($query)
    {
        return $query
            ->orderByRaw("(promoted_until IS NOT NULL AND promoted_until > NOW()) DESC")
            ->orderByDesc('promoted_rank')
            ->orderByRaw('COALESCE(bumped_at, published_at, created_at) DESC');
    }

    public function scopeWithoutBlockedContent($query, ?User $user)
    {
        if (!$user) {
            return $query;
        }

        return $query->whereDoesntHave('reports', function ($q) use ($user) {
            $q->where('user_id', $user->id);
        })->whereDoesntHave('business.blocks', function ($q) use ($user) {
            $q->where('user_id', $user->id);
        });
    }

    public function scopeSearch($query, $search)
    {
        if (empty($search)) {
            return $query;
        }

        $originalSearch = trim($search);

        // 1. Calculate relevance score using Full-Text
        $query->selectRaw("*, MATCH(title, description) AGAINST(? IN NATURAL LANGUAGE MODE) as relevance", [$originalSearch]);

        $query->where(function ($q) use ($originalSearch) {
            // Natural Language Match (Excellent for relevance ranking)
            $q->whereRaw("MATCH(title, description) AGAINST(? IN NATURAL LANGUAGE MODE)", [$originalSearch]);
            
            // Fuzzy Typo Fallback: Match characters in sequence (e.g. 'iphne' matches 'iphone')
            // Transform 'iphne' to 'i%p%h%n%e%'
            $fuzzy = '%' . implode('%', str_split(str_replace(' ', '', $originalSearch))) . '%';
            if (strlen($originalSearch) >= 3) {
                $q->orWhere('title', 'like', $fuzzy);
            }

            // Phonetic Check Fallback: First word phonetic match
            $q->orWhereRaw("SOUNDEX(SUBSTRING_INDEX(title, ' ', 1)) = SOUNDEX(?)", [$originalSearch]);
            
            // Simple partial match for safety
            $q->orWhere('title', 'like', "%{$originalSearch}%");
        });

        // The ranking will be handled by the controller to avoid conflicts with orderedForFeed
        return $query;
    }

    public static function generateUniqueSlug(string $title, ?int $ignoreId = null): string
    {
        $base = Str::slug($title);
        $slug = $base;

        $i = 1;
        while (self::withTrashed()
            ->when($ignoreId, fn($q) => $q->where('id', '!=', $ignoreId))
            ->where('slug', $slug)
            ->exists()) {
            $slug = $base . '-' . $i;
            $i++;
        }

        return $slug;
    }

    public static function generateUniqueSku(): string
    {
        do {
            $sku = 'LST-' . Str::upper(Str::random(10));
        } while (self::withTrashed()->where('sku', $sku)->exists());

        return $sku;
    }

    public static function generateUniqueBarcode(): string
    {
        do {
            $barcode = (string) random_int(1000000000000, 9999999999999);
        } while (self::withTrashed()->where('barcode', $barcode)->exists());

        return $barcode;
    }

    /**
     * Get the thumbnail URL for the listing.
     */
    public function getThumbnailUrlAttribute(): ?string
    {
        // Check for primary image first
        $primary = $this->media->where('type', 'image')->where('is_primary', true)->first();
        if ($primary) {
            return $primary->watermarked_url ?? $primary->url;
        }

        // Fallback to first image
        $first = $this->media->where('type', 'image')->first();
        if ($first) {
            return $first->watermarked_url ?? $first->url;
        }

        return null;
    }

    /**
     * Get the formatted price.
     */
    public function getPriceFormattedAttribute(): string
    {
        if ($this->price_amount === null) {
            return 'Contact for Price';
        }

        if ($this->price_amount == 0) {
            return 'Free';
        }

        return $this->currency . ' ' . number_format($this->price_amount, 0);
    }

    public function reports(): HasMany
    {
        return $this->hasMany(ListingReport::class);
    }
}
