<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class Business extends Model
{
    use SoftDeletes;

    public const VERIFICATION_UNVERIFIED = 'unverified';
    public const VERIFICATION_PENDING = 'pending';
    public const VERIFICATION_VERIFIED = 'verified';
    public const VERIFICATION_REJECTED = 'rejected';

    protected $fillable = [
        'owner_user_id',
        'name',
        'description',
        'business_type',
        'business_category',
        'status',
        'slug',
        'address',
        'district',
        'website',
        'facebook_url',
        'instagram_url',
        'x_url',
        'linkedin_url',
        'youtube_url',
        'tiktok_url',
        'logo',
        'cover',
        'primary_color',
        'secondary_color',
        'primary_phone',
        'secondary_phone',
        'primary_whatsapp',
        'secondary_whatsapp',
        'is_primary_phone_verified',
        'primary_phone_verified_at',
        'is_secondary_phone_verified',
        'secondary_phone_verified_at',
        'is_primary_whatsapp_verified',
        'primary_whatsapp_verified_at',
        'is_secondary_whatsapp_verified',
        'secondary_whatsapp_verified_at',
        'is_feedback_enabled',
        'is_chat_enabled',
        'allow_push_notifications',
        'allow_email_notifications',
        'allow_sms_notifications',
        'verification_status',
        'verification_requested_at',
        'verified_at',
        'verified_by_user_id',
        'verification_notes',
    ];

    protected function casts(): array
    {
        return [
            'owner_user_id' => 'integer',
            'verified_by_user_id' => 'integer',
            'is_feedback_enabled' => 'boolean',
            'is_chat_enabled' => 'boolean',
            'allow_push_notifications' => 'boolean',
            'allow_email_notifications' => 'boolean',
            'allow_sms_notifications' => 'boolean',
            'is_primary_phone_verified' => 'boolean',
            'is_secondary_phone_verified' => 'boolean',
            'is_primary_whatsapp_verified' => 'boolean',
            'is_secondary_whatsapp_verified' => 'boolean',
            'verification_requested_at' => 'datetime',
            'verified_at' => 'datetime',
            'primary_phone_verified_at' => 'datetime',
            'secondary_phone_verified_at' => 'datetime',
            'primary_whatsapp_verified_at' => 'datetime',
            'secondary_whatsapp_verified_at' => 'datetime',
        ];
    }

    protected static function booted(): void
    {
        static::creating(function (Business $business) {
            if (empty($business->slug)) {
                $business->slug = self::generateUniqueSlug($business->name);
            }

            if (empty($business->verification_status)) {
                $business->verification_status = self::VERIFICATION_UNVERIFIED;
            }
        });

    }

    public function owner(): BelongsTo
    {
        return $this->belongsTo(User::class, 'owner_user_id');
    }

    public function memberships(): HasMany
    {
        return $this->hasMany(BusinessMembership::class);
    }

    public function members(): BelongsToMany
    {
        return $this->belongsToMany(User::class, 'business_memberships')
            ->withPivot(['role', 'added_by_user_id', 'joined_at'])
            ->withTimestamps();
    }

    public function invitations(): HasMany
    {
        return $this->hasMany(BusinessInvitation::class);
    }

    public function followers(): BelongsToMany
    {
        return $this->belongsToMany(User::class, 'business_follows')
            ->withTimestamps();
    }

    public function feedbacks(): HasMany
    {
        return $this->hasMany(BusinessFeedback::class);
    }

    public function feedbackResponses(): HasMany
    {
        return $this->hasMany(BusinessFeedbackResponse::class);
    }

    public function listings(): HasMany
    {
        return $this->hasMany(Listing::class);
    }

    public function subscriptions(): HasMany
    {
        return $this->hasMany(UserSubscription::class);
    }

    public function activeSubscription(): ?UserSubscription
    {
        $active = $this->subscriptions()
            ->where('status', 'active')
            ->where('starts_at', '<=', now())
            ->where('expires_at', '>', now())
            ->latest('expires_at')
            ->first();

        if (!$active) {
            $freePackage = PremiumPackage::where('slug', 'free')->first();
            if ($freePackage) {
                $sub = new UserSubscription([
                    'business_id' => $this->id,
                    'package_id' => $freePackage->id,
                    'status' => 'active',
                    'starts_at' => $this->created_at ?? now(),
                    'expires_at' => now()->addYears(10),
                ]);
                $sub->setRelation('package', $freePackage);
                return $sub;
            }
        }

        return $active;
    }

    public function isVerifiedMember(): bool
    {
        $sub = $this->activeSubscription();
        return (bool) ($sub?->package?->verified_badge);
    }

    public function hasAnalyticsAccess(): bool
    {
        $sub = $this->activeSubscription();
        return (bool) ($sub?->package?->analytics_access);
    }

    public function shouldRemoveAds(): bool
    {
        $sub = $this->activeSubscription();
        return (bool) ($sub?->package?->remove_ads);
    }

    public function conversations(): HasMany
    {
        return $this->hasMany(Conversation::class);
    }

    public function getLogoUrlAttribute(): ?string
    {
        if (!$this->logo) {
            return null;
        }

        if (filter_var($this->logo, FILTER_VALIDATE_URL)) {
            return $this->logo;
        }

        return Storage::disk('public')->url($this->logo);
    }

    public function getCoverUrlAttribute(): ?string
    {
        if (!$this->cover) {
            return null;
        }

        if (filter_var($this->cover, FILTER_VALIDATE_URL)) {
            return $this->cover;
        }

        return Storage::disk('public')->url($this->cover);
    }

    public function isVerified(): bool
    {
        return $this->verification_status === self::VERIFICATION_VERIFIED;
    }

    public function requestVerification(): void
    {
        if ($this->verification_status === self::VERIFICATION_VERIFIED) {
            return;
        }

        $this->update([
            'verification_status' => self::VERIFICATION_PENDING,
            'verification_requested_at' => now(),
        ]);
    }

    public static function generateUniqueSlug(string $name, ?int $ignoreId = null): string
    {
        $base = Str::slug($name);
        $slug = $base;

        $i = 1;
        while (self::query()
            ->when($ignoreId, fn($q) => $q->where('id', '!=', $ignoreId))
            ->where('slug', $slug)
            ->exists()) {
            $slug = $base . '-' . $i;
            $i++;
        }

        return $slug;
    }

    public function isFavoritedBy(?User $user): bool
    {
        if (!$user) {
            return false;
        }

        return $this->followers()->where('user_id', $user->id)->exists();
    }

    public function blocks(): HasMany
    {
        return $this->hasMany(BusinessBlock::class);
    }

    /**
     * Check if a user is a member (or owner) of this business.
     */
    public function isMember(?User $user): bool
    {
        if (!$user) {
            return false;
        }

        if ((int) $this->owner_user_id === (int) $user->id) {
            return true;
        }

        return $this->members()->where('user_id', $user->id)->exists();
    }
}
