<?php

namespace App\Http\Resources;

use Illuminate\Http\Request;
use Illuminate\Http\Resources\Json\JsonResource;

class UserResource extends JsonResource
{
    /**
     * Transform the resource into an array.
     *
     * @return array<string, mixed>
     */
    public function toArray(Request $request): array
    {
        return [
            // Identifiers
            'id' => $this->id,
            'uuid' => $this->uuid,
            
            // Profile
            'full_name' => $this->full_name,
            'username' => $this->username,
            'bio' => $this->bio,
            'profile_photo_url' => $this->profile_photo_url,
            'date_of_birth' => $this->date_of_birth?->format('Y-m-d'),
            'gender' => $this->gender,
            
            // Contact (conditionally shown based on privacy settings)
            'email' => $this->when($this->shouldShowEmail($request), $this->email),
            'phone' => $this->when($this->shouldShowPhone($request), $this->phone),
            'secondary_phone' => $this->when($this->shouldShowPhone($request), $this->secondary_phone),
            'whatsapp_number' => $this->when($this->shouldShowPhone($request), $this->whatsapp_number),
            'secondary_whatsapp_number' => $this->when($this->shouldShowPhone($request), $this->secondary_whatsapp_number),
            
            // Verification Status
            'is_email_verified' => $this->is_email_verified,
            'is_phone_verified' => $this->is_phone_verified,
            'identity_verification_status' => $this->identity_verification_status,

            'contact_verification' => $this->when($this->isOwnProfile($request), [
                'primary_phone_verified' => $this->is_phone_verified || !is_null($this->phone_verified_at),
                'secondary_phone_verified' => !is_null($this->secondary_phone_verified_at),
                'primary_whatsapp_verified' => !is_null($this->whatsapp_verified_at),
                'secondary_whatsapp_verified' => !is_null($this->secondary_whatsapp_verified_at),
            ]),
            
            // Location (conditionally shown based on privacy settings)
            'location' => $this->when($this->shouldShowLocation($request), [
                'country' => $this->country,
                'country_code' => $this->country_code,
                'state' => $this->state,
                'city' => $this->city,
                'district' => $this->district,
                'address' => $this->address,
                'postal_code' => $this->postal_code,
                'coordinates' => $this->coordinates,
                'timezone' => $this->timezone,
                'full_location' => $this->full_location,
            ]),
            
            // Social Links
            'social_links' => [
                'website' => $this->website,
                'facebook' => $this->facebook_url,
                'twitter' => $this->twitter_url,
                'instagram' => $this->instagram_url,
                'linkedin' => $this->linkedin_url,
                'tiktok' => $this->tiktok_url,
                'youtube' => $this->youtube_url,
            ],
            
            // Account Status
            'status' => $this->status,
            'role' => $this->role,
            
            // Preferences (only for own profile)
            'preferences' => $this->when($this->isOwnProfile($request), [
                'language' => $this->language,
                'currency' => $this->currency,
                'notifications' => [
                    'enabled' => $this->is_notifications_enabled,
                    'email' => $this->is_email_notifications_enabled,
                    'sms' => $this->is_sms_notifications_enabled,
                    'push' => $this->is_push_notifications_enabled,
                    'marketing' => $this->is_marketing_emails_enabled,
                ],
                'privacy' => [
                    'show_phone_publicly' => $this->show_phone_publicly,
                    'show_email_publicly' => $this->show_email_publicly,
                    'show_location_publicly' => $this->show_location_publicly,
                ],
            ]),
            
            // Security (only for own profile)
            'security' => $this->when($this->isOwnProfile($request), [
                'two_factor_enabled' => $this->two_factor_enabled,
                'password_changed_at' => $this->password_changed_at?->toIso8601String(),
                'last_login_at' => $this->last_login_at?->toIso8601String(),
            ]),
            
            // Referral
            'referral_code' => $this->referral_code,
            'referral_count' => $this->referral_count,
            
            // Timestamps
            'created_at' => $this->created_at?->toIso8601String(),
            'updated_at' => $this->updated_at?->toIso8601String(),
        ];
    }

    /**
     * Check if this is the user's own profile.
     */
    private function isOwnProfile(Request $request): bool
    {
        return $request->user()?->id === $this->id;
    }

    /**
     * Check if email should be shown.
     */
    private function shouldShowEmail(Request $request): bool
    {
        if ($this->isOwnProfile($request)) {
            return true;
        }
        if ($request->user()?->isAdmin()) {
            return true;
        }
        return (bool) $this->show_email_publicly;
    }

    /**
     * Check if phone should be shown.
     */
    private function shouldShowPhone(Request $request): bool
    {
        if ($this->isOwnProfile($request)) {
            return true;
        }
        if ($request->user()?->isAdmin()) {
            return true;
        }
        return (bool) $this->show_phone_publicly;
    }

    /**
     * Check if location should be shown.
     */
    private function shouldShowLocation(Request $request): bool
    {
        if ($this->isOwnProfile($request)) {
            return true;
        }
        if ($request->user()?->isAdmin()) {
            return true;
        }
        return (bool) $this->show_location_publicly;
    }
}
