<?php

namespace App\Http\Controllers;

use App\Models\Listing;
use App\Models\Business;
use App\Models\ListingMedia;
use App\Models\User;
use App\Models\ListingReport;
use App\Models\BusinessBlock;
use App\Jobs\AutoReviewListing;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use App\Mail\AdReported;
use App\Services\ImageProcessingService;

class WebListingController extends Controller
{
    public function show(string $slug)
    {
        $listing = Listing::where('slug', $slug)
            ->withoutBlockedContent(auth('web')->user())
            ->with(['media', 'createdBy', 'business' => function($query) {
                $query->withCount('feedbacks');
            }])
            ->firstOrFail();

        // If not published, only owner can view
        if ($listing->status !== Listing::STATUS_PUBLISHED) {
            $isOwner = auth('web')->check() && auth('web')->id() == $listing->created_by_user_id;

            if (!$isOwner) {
                abort(404);
            }
        }

        // Increment view count
        $listing->increment('views_count');

        $relatedListings = Listing::where('category', $listing->category)
            ->where('id', '!=', $listing->id)
            ->where('status', 'published')
            ->with('media')
            ->limit(4)
            ->get();

        return view('web.ad-details', [
            'listing' => $listing,
            'relatedListings' => $relatedListings,
        ]);
    }

    public function report(Request $request, $slug)
    {
        $listing = Listing::where('slug', $slug)->firstOrFail();

        $validated = $request->validate([
            'reason' => 'required|string|max:255',
            'explanation' => 'nullable|string|max:1000',
        ]);

        $data = [
            'reason' => $validated['reason'],
            'explanation' => $validated['explanation'] ?? '',
            'listing_title' => $listing->title,
            'listing_url' => route('listing.show', $slug),
            'reporter_ip' => $request->ip(),
            'reporter_email' => $request->user()?->email ?? 'Anonymous',
            'reported_at' => now(),
        ];

        // Save to Database if user is logged in
        if (auth('web')->check()) {
            ListingReport::firstOrCreate(
                [
                    'listing_id' => $listing->id,
                    'user_id' => auth('web')->id(),
                ],
                [
                    'reason' => $validated['reason'],
                    'explanation' => $validated['explanation'] ?? '',
                ]
            );
        }

        // Send Email
        $recipient = env('REPORTING_EMAIL');
        if($recipient) {
            Mail::to($recipient)->queue(new \App\Mail\AdReported($data));
        }

        return response()->json(['message' => 'Report submitted successfully. We will review it shortly.']);
    }

    public function blockBusiness(Request $request, $slug)
    {
        $business = Business::where('slug', $slug)->firstOrFail();

        if (!auth('web')->check()) {
            return response()->json(['message' => 'You must be logged in to block a business.'], 401);
        }

        BusinessBlock::firstOrCreate([
            'business_id' => $business->id,
            'user_id' => auth('web')->id(),
        ]);

        return response()->json(['message' => 'Business blocked. You will no longer see their listings.']);
    }
    public function hotDeals()
    {
        $listings = Listing::publicVisible()
            ->withoutBlockedContent(auth('web')->user())
            ->with(['media', 'createdBy', 'business'])
            ->inRandomOrder()
            ->take(50)
            ->get();

        return view('web.hot-deals', compact('listings'));
    }

    public function search(Request $request)
    {
        $query = Listing::publicVisible()
            ->withoutBlockedContent(auth('web')->user())
            ->with(['media', 'createdBy', 'business']);

        // Text search
        $searchQuery = $request->input('q');
        $suggestion = null;

        if ($request->filled('q')) {
            $query->search($searchQuery);
        }

        // Location filter
        if ($request->filled('location') && $request->input('location') !== 'All Uganda') {
            $query->where('location_district', $request->input('location'));
        }



        // Category filter
        if ($request->filled('category')) {
            $query->where('category', $request->input('category'));
        }

        // Subcategory filter
        if ($request->filled('subcategory')) {
            $query->where('sub_category', $request->input('subcategory'));
        }

        // Custom fields filtering
        if ($request->filled('custom') && is_array($request->input('custom'))) {
            foreach ($request->input('custom') as $key => $value) {
                if (!empty($value)) {
                    // Check if value is comma-separated (from checkboxes)
                    if (str_contains($value, ',')) {
                        $values = explode(',', $value);
                        $query->where(function ($q) use ($key, $values) {
                            foreach ($values as $v) {
                                $q->orWhereJsonContains('custom_fields->' . $key, $v);
                            }
                        });
                    } else {
                        $query->whereJsonContains('custom_fields->' . $key, $value);
                    }
                }
            }
        }

        // Price filters
        if ($request->filled('min_price')) {
            $query->where('price_amount', '>=', (float) $request->input('min_price'));
        }
        if ($request->filled('max_price')) {
            $query->where('price_amount', '<=', (float) $request->input('max_price'));
        }

        // Sorting
        $sort = $request->input('sort', 'relevance');
        if ($request->filled('q') && $sort === 'relevance') {
            $query->orderBy('relevance', 'desc');
        } else {
            switch ($sort) {
                case 'price_asc':
                    $query->orderBy('price_amount', 'asc');
                    break;
                case 'price_desc':
                    $query->orderBy('price_amount', 'desc');
                    break;
                case 'newest':
                    $query->orderBy('created_at', 'desc');
                    break;
                default:
                    $query->orderBy('bumped_at', 'desc')->orderBy('created_at', 'desc');
            }
        }

        $listings = $query->paginate(24)->withQueryString();

        // Track the search query (only on first page)
        if ($searchQuery && $request->input('page', 1) == 1) {
            try {
                \App\Models\SearchQuery::create([
                    'query' => strtolower($searchQuery),
                    'search_type' => 'listing',
                    'ip_address' => $request->ip(),
                    'user_agent' => $request->userAgent(),
                    'results_count' => $listings->total(),
                    'searched_at' => now(),
                    'user_id' => auth('web')->id(),
                ]);
            } catch (\Exception $e) {
                \Log::error('Failed to log search query: ' . $e->getMessage());
            }
        }

        // Check for "Did you mean?" suggestions
        if ($searchQuery && $listings->total() > 0) {
            $topListing = $listings->first();
            // If relevance is 0 or very low, it means it was a fuzzy/Soundex match
            if (!$topListing->relevance || $topListing->relevance < 1.0) {
                $titleWords = explode(' ', strtolower($topListing->title));
                $searchLower = strtolower($searchQuery);
                
                foreach ($titleWords as $word) {
                    $word = preg_replace('/[^a-z0-9]/', '', $word);
                    if (strlen($word) < 3) continue;
                    
                    if ($word[0] == $searchLower[0] && abs(strlen($word) - strlen($searchLower)) <= 2) {
                        $suggestion = $word;
                        break;
                    }
                }
            }
        }

        // Load categories for filter dropdown
        $categoriesPath = public_path('assets/top_level_categories.json');
        $categories = file_exists($categoriesPath) 
            ? json_decode(file_get_contents($categoriesPath), true) 
            : [];

        // Load subcategories if category is selected
        $subcategories = [];
        if ($request->filled('category')) {
            $allSubcategoriesPath = public_path('assets/categories.json');
            if (file_exists($allSubcategoriesPath)) {
                $allSubcategories = json_decode(file_get_contents($allSubcategoriesPath), true);
                
                // Find parent ID for current category
                $parentCat = collect($categories)->firstWhere('Category', $request->input('category'));
                if ($parentCat) {
                    $parentId = $parentCat['id'];
                    $subcategories = collect($allSubcategories)->where('parent', $parentId)->values()->all();
                }
            }
        }

        // Load custom fields if subcategory is selected
        $customFields = [];
        if ($request->filled('subcategory')) {
            $typesPath = public_path('assets/types.json');
            if (file_exists($typesPath)) {
                $typesData = json_decode(file_get_contents($typesPath), true);
                
                // Need to find subcategory ID
                $allSubcategoriesPath = public_path('assets/categories.json');
                if (file_exists($allSubcategoriesPath)) {
                    $allSubcategories = json_decode(file_get_contents($allSubcategoriesPath), true);
                    $subCat = collect($allSubcategories)->firstWhere('Category', $request->input('subcategory'));
                    
                    if ($subCat && isset($typesData['customFields'][$subCat['id']])) {
                        $customFields = $typesData['customFields'][$subCat['id']];
                    }
                }
            }
        }

        // Load popular searches
        $popularSearches = \App\Models\SearchQuery::select('query')
            ->selectRaw('count(*) as count')
            ->whereNotNull('query')
            ->groupBy('query')
            ->orderByDesc('count')
            ->limit(10)
            ->pluck('query')
            ->all();

        return view('web.search', [
            'listings' => $listings,
            'categories' => $categories,
            'subcategories' => $subcategories,
            'customFields' => $customFields,
            'popularSearches' => $popularSearches,
            'filters' => array_merge([
                'q' => '',
                'location' => '',
                'category' => '',
                'subcategory' => '',
                'min_price' => '',
                'max_price' => '',
                'sort' => 'relevance',
                'custom' => []
            ], $request->only(['q', 'location', 'category', 'subcategory', 'min_price', 'max_price', 'sort', 'custom'])),
            'suggestion' => $suggestion,
        ]);
    }

    public function create()
    {
        $user = auth()->user();

        // Redirect to business creation if the user has no shop
        if ($user->allBusinesses()->count() === 0) {
            return redirect()->route('business.create')
                ->with('info', 'Please create your shop profile before posting your first ad.');
        }

        $selectedBusinessId = session('selected_business_id');
        
        $business = null;
        if ($selectedBusinessId) {
            $business = $user->allBusinesses()->where('id', $selectedBusinessId)->first();
        }

        if (!$business) {
            $business = $user->allBusinesses()->first();
        }

        return view('web.post-ad', compact('business'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required|min:5|max:100',
            'description' => 'required|min:20',
            'price' => 'required|numeric',
            'category' => 'required',
            'sub_category' => 'required',
            'location' => 'required',
            'media_ids' => 'required_without:photos|array|min:1',
            'photos' => 'required_without:media_ids|array|min:1',
            'photos.*' => 'image|mimes:jpeg,png,jpg,webp|max:5120', // 5MB max
        ]);

        // Ensure user is at least a seller
        if (auth()->user()->role === User::ROLE_USER) {
            auth()->user()->update(['role' => User::ROLE_SELLER]);
        }

        // Get or create business
        $selectedBusinessId = session('selected_business_id');
        $business = null;
        
        if ($selectedBusinessId) {
            $business = auth()->user()->allBusinesses()->where('id', $selectedBusinessId)->first();
        }

        if (!$business) {
            $business = auth()->user()->allBusinesses()->first();
        }

        if (!$business) {
            $business = Business::create([
                'owner_user_id' => auth()->id(),
                'name' => auth()->user()->full_name . "'s Shop",
                'slug' => Str::slug(auth()->user()->full_name . "-shop-" . rand(100, 999)),
                'status' => 'active',
            ]);
        }

        $listing = Listing::create([
            'created_by_user_id' => auth()->id(),
            'business_id' => $business->id,
            'title' => $request->title,
            'description' => $request->description,
            'price_amount' => $request->price,
            'purchase_price_amount' => $request->purchase_price,
            'quantity' => 0,
            'currency' => 'UGX',
            'status' => 'review',
            'category' => $request->category,
            'sub_category' => $request->sub_category,
            'sub_sub_category' => $request->sub_sub_category,
            'location_district' => $request->location,
            'custom_fields' => $request->custom_fields,
            'allow_pos' => $request->has('allow_pos'),
        ]);

        // Process pre-uploaded photos via media_ids
        if ($request->filled('media_ids')) {
            $mediaIds = (array) $request->input('media_ids');
            
            $preUploadedMedia = ListingMedia::query()
                ->whereIn('id', $mediaIds)
                ->whereNull('listing_id')
                ->where('uploaded_by_user_id', auth()->id())
                ->get();

            foreach ($preUploadedMedia as $index => $media) {
                $media->update([
                    'listing_id' => $listing->id,
                    'sort_order' => $index,
                    'is_primary' => $index === 0,
                ]);
            }
        }
        // Fallback: Process direct photo uploads
        elseif ($request->hasFile('photos')) {
            foreach ($request->file('photos') as $index => $photo) {
                $path = $photo->store('listings', 'public');
                
                $listing->media()->create([
                    'path' => $path,
                    'type' => ListingMedia::TYPE_IMAGE,
                    'is_primary' => $index === 0,
                    'uploaded_by_user_id' => auth()->id(),
                    'size_bytes' => $photo->getSize(),
                    'mime_type' => $photo->getMimeType(),
                ]);
            }
        }

        // Trigger Auto-Review (delayed)
        if ($listing->status === Listing::STATUS_REVIEW) {
            AutoReviewListing::dispatch($listing->id)->delay(now()->addMinutes(rand(1, 2)));
        }

        return redirect()->route('listing.show', $listing->slug)->with('success', 'Ad posted successfully! It is currently under review.');
    }

    public function repost($id)
    {
        $listing = Listing::findOrFail($id);

        if ($listing->created_by_user_id != auth()->id()) {
            abort(403);
        }

        $listing->update([
            'status' => 'review',
            'created_at' => now(), // Refresh creation date to appear new
            'published_at' => null, // Reset published at until approved
            'bumped_at' => null,
            'expires_at' => null,
        ]);

        if (request()->ajax()) {
            return response()->json([
                'success' => true,
                'message' => 'Ad reposted for review! It will be active after approval.'
            ]);
        }

        return back()->with('success', 'Ad reposted for review! It will be active after approval.');
    }

    public function close($id)
    {
        $listing = Listing::findOrFail($id);

        if ($listing->created_by_user_id != auth()->id()) {
            abort(403);
        }

        $listing->update([
            'status' => 'draft',
        ]);

        if (request()->ajax()) {
            return response()->json([
                'success' => true,
                'message' => 'Ad closed successfully. It is now hidden from public view.'
            ]);
        }

        return back()->with('success', 'Ad closed successfully. It is now hidden from public view.');
    }

    public function destroy($id)
    {
        $listing = Listing::findOrFail($id);
        
        if ($listing->created_by_user_id != auth()->id()) {
            abort(403);
        }

        $listing->delete();

        if (request()->ajax()) {
            return response()->json([
                'success' => true,
                'message' => 'Ad deleted successfully.'
            ]);
        }

        return back()->with('success', 'Ad deleted successfully.');
    }

    public function bulkRepost(Request $request)
    {
        $request->validate([
            'ids' => 'required|array',
            'ids.*' => 'exists:listings,id',
        ]);

        $ids = $request->input('ids');
        
        $count = Listing::whereIn('id', $ids)
            ->where('created_by_user_id', auth()->id())
            ->update([
                'status' => 'review',
                'created_at' => now(),
                'published_at' => null,
                'bumped_at' => null,
                'expires_at' => null,
            ]);

        if (request()->ajax()) {
            return response()->json([
                'success' => true,
                'message' => "$count ads submitted for review!"
            ]);
        }

        return back()->with('success', "$count ads submitted for review!");
    }

    public function bulkDestroy(Request $request)
    {
        $request->validate([
            'ids' => 'required|array',
            'ids.*' => 'exists:listings,id',
        ]);

        $ids = $request->input('ids');
        
        $count = Listing::whereIn('id', $ids)
            ->where('created_by_user_id', auth()->id())
            ->delete();

        if (request()->ajax()) {
            return response()->json([
                'success' => true,
                'message' => "$count ads deleted successfully."
            ]);
        }

        return back()->with('success', "$count ads deleted successfully.");
    }

    public function uploadMedia(Request $request)
    {
        $request->validate([
            'media' => 'required|array|max:8',
            'media.*' => 'image|mimes:jpeg,png,jpg,webp|max:5120', // 5MB max
        ]);

        $user = auth()->user();
        $uploadedMedia = [];
        $imageService = new ImageProcessingService();
        
        $selectedBusinessId = session('selected_business_id');
        $business = null;
        if ($selectedBusinessId) {
            $business = auth()->user()->allBusinesses()->where('id', $selectedBusinessId)->first();
        }
        if (!$business) {
            $business = auth()->user()->allBusinesses()->first();
        }
        
        $businessName = $business ? $business->name : $user->full_name;

        foreach ($request->file('media') as $file) {
            if (!$file) continue;

            $mime = (string) $file->getClientMimeType();
            $path = $file->store('listing-media', 'public');
            $contentHash = hash_file('sha256', $file->getRealPath());

            $watermarkedPath = null;
            $tags = [];
            
            // Process images (compress + watermark + tag detection)
            try {
                $processed = $imageService->processImage($path, $businessName);
                $path = $processed['compressed_path'];
                $watermarkedPath = $processed['watermarked_path'];
                $tags = $processed['tags'] ?? [];
            } catch (\Exception $e) {
                // If processing fails, use original xpath
                \Log::error('Image processing failed: ' . $e->getMessage());
            }

            $media = ListingMedia::create([
                'listing_id' => null,
                'uploaded_by_user_id' => $user->id,
                'path' => $path,
                'watermarked_path' => $watermarkedPath,
                'tags' => $tags,
                'mime_type' => $mime,
                'size_bytes' => $file->getSize(),
                'content_hash' => $contentHash,
                'type' => ListingMedia::TYPE_IMAGE,
                'sort_order' => 0,
                'is_primary' => false,
            ]);

            $uploadedMedia[] = [
                'id' => $media->id,
                'url' => Storage::disk('public')->url($path),
            ];
        }

        return response()->json([
            'success' => true,
            'media' => $uploadedMedia,
        ]);
    }

    public function edit(string $slug)
    {
        $listing = Listing::where('slug', $slug)
            ->with(['media', 'business'])
            ->firstOrFail();

        // Check ownership
        if ($listing->created_by_user_id != auth()->id()) {
            abort(403, 'Unauthorized action.');
        }

        $business = $listing->business;

        return view('web.edit-ad', compact('listing', 'business'));
    }

    public function update(Request $request, $id)
    {
        $listing = Listing::findOrFail($id);

        // Check ownership
        if ($listing->created_by_user_id != auth()->id()) {
            abort(403);
        }

        $request->validate([
            'title' => 'required|min:5|max:100',
            'description' => 'required|min:20',
            'price' => 'required|numeric',
            'category' => 'required',
            'sub_category' => 'required',
            'location' => 'required',
            'media_ids' => 'sometimes|array',
        ]);

        $needsReview = false;
        
        // Significant fields mapping
        $reviewFields = [
            'title' => $request->title,
            'description' => $request->description,
            'price_amount' => $request->price,
            'category' => $request->category,
            'sub_category' => $request->sub_category,
            'sub_sub_category' => $request->sub_sub_category,
            'location_district' => $request->location,
        ];

        foreach ($reviewFields as $field => $newValue) {
            if ($listing->$field != $newValue) {
                $needsReview = true;
                break;
            }
        }

        // Also check media and custom fields
        if (!$needsReview) {
            if ($request->filled('media_ids')) {
                $needsReview = true;
            } elseif (json_encode($listing->custom_fields) !== json_encode($request->custom_fields)) {
                $needsReview = true;
            }
        }

        $updateData = [
            'title' => $request->title,
            'description' => $request->description,
            'price_amount' => $request->price,
            'purchase_price_amount' => $request->purchase_price ?? $listing->purchase_price_amount,
            'quantity' => $request->quantity ?? $listing->quantity,
            'category' => $request->category,
            'sub_category' => $request->sub_category,
            'sub_sub_category' => $request->sub_sub_category,
            'location_district' => $request->location,
            'custom_fields' => $request->custom_fields,
            'allow_pos' => $request->has('allow_pos'),
        ];

        if ($needsReview) {
            $updateData['status'] = 'review';
        }

        $listing->update($updateData);

        // Process NEW pre-uploaded photos via media_ids
        if ($request->filled('media_ids')) {
            $mediaIds = (array) $request->input('media_ids');
            
            $nextSortOrder = $listing->media()->max('sort_order') + 1;

            $preUploadedMedia = ListingMedia::query()
                ->whereIn('id', $mediaIds)
                ->whereNull('listing_id')
                ->where('uploaded_by_user_id', auth()->id())
                ->get();

            foreach ($preUploadedMedia as $index => $media) {
                $media->update([
                    'listing_id' => $listing->id,
                    'sort_order' => $nextSortOrder + $index,
                    'is_primary' => $listing->media()->where('is_primary', true)->count() === 0 && $index === 0,
                ]);
            }
        }

        if (($needsReview || $listing->wasChanged('status')) && $listing->status === Listing::STATUS_REVIEW) {
            AutoReviewListing::dispatch($listing->id)->delay(now()->addMinutes(rand(1, 2)));
        }

        return redirect()->route('listing.show', $listing->slug)->with('success', 'Ad updated successfully! It is currently under review.');
    }

    public function deleteMedia(Request $request, $mediaId)
    {
        $media = ListingMedia::findOrFail($mediaId);

        // Check ownership of the media via its associated listing OR the uploader
        if ($media->uploaded_by_user_id != auth()->id()) {
            abort(403);
        }

        // Keep at least one photo if it's already attached to a listing
        if ($media->listing_id) {
            $listing = $media->listing;
            if ($listing->media()->count() <= 1) {
                return response()->json([
                    'success' => false,
                    'message' => 'At least one photo is required.'
                ], 422);
            }

            // If we are deleting the primary image, make the next one primary
            if ($media->is_primary) {
                $nextPrimary = $listing->media()->where('id', '!=', $mediaId)->orderBy('sort_order')->first();
                if ($nextPrimary) {
                    $nextPrimary->update(['is_primary' => true]);
                }
            }
        }

        // Delete file and record
        Storage::disk('public')->delete($media->path);
        if ($media->watermarked_path) {
            Storage::disk('public')->delete($media->watermarked_path);
        }
        $media->delete();

        return response()->json(['success' => true]);
    }
}
