<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Business;
use App\Models\User;

class WebBusinessController extends Controller
{
    public function show(Request $request, $slug)
    {
        $business = Business::with(['owner'])
            ->withCount(['followers', 'listings', 'feedbacks'])
            ->where('slug', $slug)
            ->firstOrFail();

        $listings = $business->listings()
            ->with(['media', 'business', 'createdBy']) // Eager load for cards
            ->publicVisible()
            ->when($request->q, function($query, $q) {
                return $query->where('title', 'like', '%'.$q.'%');
            })
            ->latest()
            ->paginate(12)
            ->withQueryString(); // Keep search query in pagination links

        $feedbacks = $business->feedbacks()
            ->with(['user', 'responses', 'attachments'])
            ->where('moderation_status', 'approved') // Assuming there's moderation
            ->latest()
            ->limit(20)
            ->get();

        return view('web.business-details', compact('business', 'listings', 'feedbacks'));
    }

    /**
     * Show the form for creating a new shop.
     */
    public function create()
    {
        $districtsPath = public_path('assets/districts.json');
        $districts = file_exists($districtsPath) 
            ? json_decode(file_get_contents($districtsPath), true) 
            : [];
            
        return view('web.business-create', compact('districts'));
    }

    /**
     * Store a newly created shop in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'business_type' => 'required|string|max:255',
            'business_category' => 'required|string|max:255',
            'description' => 'nullable|string',
            'primary_phone' => 'required|string|max:20',
            'primary_whatsapp' => 'required|string|max:20',
            'district' => 'required|string|max:100',
            'logo' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'cover' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:5120',
        ]);

        // Format phone numbers to +256 standard
        $validated['primary_phone'] = $this->formatUgPhoneNumber($validated['primary_phone']);
        if (!empty($validated['primary_whatsapp'])) {
            $validated['primary_whatsapp'] = $this->formatUgPhoneNumber($validated['primary_whatsapp']);
        }

        $business = new Business($validated);
        $business->owner_user_id = auth()->id();
        $business->status = 'active'; // Ensure it's active

        // Handle Logo Upload
        if ($request->hasFile('logo')) {
            $path = $request->file('logo')->store('businesses/logos', 'public');
            $business->logo = $path;
        }

        // Handle Cover Upload
        if ($request->hasFile('cover')) {
            $path = $request->file('cover')->store('businesses/covers', 'public');
            $business->cover = $path;
        }

        $business->save();

        // Ensure user is at least a seller
        $user = auth()->user();
        if ($user->role === User::ROLE_USER) {
            $user->update(['role' => User::ROLE_SELLER]);
        }

        // Automatically switch to the new shop
        session(['selected_business_id' => $business->id]);

        return redirect()->route('business.success', $business->slug);
    }

    /**
     * Format phone number to +256 standard.
     */
    private function formatUgPhoneNumber(string $phone): string
    {
        // Remove all non-digits
        $digits = preg_replace('/[^0-9]/', '', $phone);

        // Remove leading 256 if present
        if (strpos($digits, '256') === 0) {
            $digits = substr($digits, 3);
        }

        // Remove leading 0 if present
        if (strpos($digits, '0') === 0) {
            $digits = substr($digits, 1);
        }

        return '+256' . $digits;
    }

    /**
     * Show the success page after creating a shop.
     */
    public function success($businessSlug)
    {
        $business = Business::where('slug', $businessSlug)->firstOrFail();

        // Ensure the current user owns this business
        if ($business->owner_user_id != auth()->id()) {
            abort(403);
        }

        return view('web.business-success', compact('business'));
    }
}
