<?php

namespace App\Http\Controllers\Web;

use App\Http\Controllers\Controller;
use App\Models\Business;
use App\Models\Conversation;
use App\Models\Listing;
use App\Models\Message;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\View\View;

class ChatController extends Controller
{
    /**
     * Show the chat inbox.
     */
    public function inbox(): View
    {
        return view('web.chat.inbox');
    }

    /**
     * Show a specific conversation.
     */
    public function show(Conversation $conversation): View
    {
        $user = auth('web')->user();

        if (!$this->canAccessConversation($user, $conversation)) {
            abort(403);
        }

        // Load necessary relationships for the initial view
        $conversation->load(['listing.media', 'buyer', 'business']);

        return view('web.chat.show', compact('conversation'));
    }

    public function startChat(Listing $listing, Request $request)
    {
        $user = auth('web')->user();

        // Prevent chatting with self (optional, but good UX)
        if ($listing->created_by_user_id == $user->id) {
            return back()->with('error', 'You cannot chat with yourself.');
        }

        $conversation = Conversation::firstOrCreate([
            'listing_id' => $listing->id,
            'buyer_id' => $user->id,
            'business_id' => $listing->business_id,
        ]);
        
        // Pass initial message if provided
        $params = [];
        if ($request->has('message')) {
            $params['initial_message'] = $request->message;
        }

        return redirect()->route('web.chat.show', array_merge(['conversation' => $conversation->id], $params));
    }

    // ==========================================
    // AJAX Endpoints
    // ==========================================

    /**
     * Get conversations (Buying vs Selling).
     */
    public function getConversations(Request $request)
    {
        $user = auth('web')->user();
        $type = $request->query('type', 'buying'); // 'buying' or 'selling'

        $query = Conversation::query()
            ->with(['listing.media', 'listing.business', 'buyer', 'business', 'latestMessage'])
            ->orderByDesc('last_message_at');

        if ($type === 'selling') {
            // Get conversations for businesses where the user is a member/owner
            $businessIds = Business::where('owner_user_id', $user->id)
                ->orWhereHas('memberships', function ($q) use ($user) {
                    $q->where('user_id', $user->id);
                })
                ->pluck('id');

            $query->whereIn('business_id', $businessIds);
        } else {
            // Default: Buying conversations
            $query->where('buyer_id', $user->id);
        }

        $conversations = $query->paginate(20);

        // Transform collection slightly to make it easier for JS
        $items = $conversations->getCollection()->map(function ($conversation) use ($user, $type) {
            $latest = $conversation->latestMessage;
            
            // Determine display name and avatar
            if ($type === 'buying') {
                // User is buyer, show business info
                $name = $conversation->business->name;
                $logo = $conversation->business->logo_url;
                $isVerified = $conversation->business->is_verified;
            } else {
                // User is seller, show buyer info
                $name = $conversation->buyer->full_name ?? $conversation->buyer->username ?? 'Guest';
                $logo = $conversation->buyer->profile_photo_url;
                $isVerified = false;
            }

            return [
                'id' => $conversation->id,
                'name' => $name,
                'logo' => $logo,
                'is_verified' => $isVerified,
                'listing_title' => $conversation->listing->title ?? 'Unknown Listing',
                'listing_image' => $conversation->listing->media->first()?->url ?? null,
                'last_message' => $latest ? ($latest->type === 'image' ? '📷 Photo' : \Str::limit($latest->message, 30)) : 'No messages yet',
                'last_message_at' => $conversation->last_message_at?->diffForHumans() ?? '',
                 // Logic for unread count: Count messages NOT from me AND unread
                'unread_count' => $conversation->messages()
                    ->where('sender_id', '!=', $user->id)
                    ->whereNull('read_at')
                    ->count(),
                'url' => route('web.chat.show', $conversation->id),
            ];
        });

        return response()->json([
            'conversations' => $items,
            'has_more' => $conversations->hasMorePages(),
        ]);
    }

    /**
     * Get messages for a conversation.
     */
    public function getMessages(Conversation $conversation, Request $request)
    {
        $user = auth('web')->user();

        if (!$this->canAccessConversation($user, $conversation)) {
            return response()->json(['error' => 'Unauthorized'], 403);
        }

        $messages = $conversation->messages()
            ->with('sender')
            ->orderByDesc('created_at')
            ->paginate(50);

        // Mark as read immediately when fetching (similar to mobile logic)
        $conversation->messages()
            ->where('sender_id', '!=', $user->id)
            ->whereNull('read_at')
            ->update(['read_at' => now()]);

        $items = $messages->getCollection()->map(function ($msg) use ($user) {
            return [
                'id' => $msg->id,
                'message' => $msg->message,
                'attachment' => $msg->attachment ? asset('storage/' . $msg->attachment) : null,
                'type' => $msg->type,
                'is_me' => $msg->sender_id == $user->id,
                'sender_name' => $msg->sender->full_name ?? $msg->sender->username,
                'sender_avatar' => $msg->sender->profile_photo_url,
                'created_at' => $msg->created_at->format('g:i A'), // 2:30 PM
                'created_at_raw' => $msg->created_at->toIso8601String(),
                'date_human' => $msg->created_at->diffForHumans(),
                'read_at' => $msg->read_at,
            ];
        });

        return response()->json([
            'messages' => $items,
            'next_page_url' => $messages->nextPageUrl(),
        ]);
    }

    /**
     * Send a message.
     */
    public function sendMessage(Request $request)
    {
        $user = auth('web')->user();

        $validator = Validator::make($request->all(), [
            'listing_id' => 'nullable|exists:listings,id',
            'conversation_id' => 'nullable|exists:conversations,id',
            'message' => 'nullable|string|max:2000',
            'attachment' => 'nullable|image|max:10240',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        $conversationId = $request->conversation_id;
        $listingId = $request->listing_id;

        if (!$conversationId && !$listingId) {
            return response()->json(['error' => 'Conversation ID or Listing ID required'], 422);
        }

        if ($conversationId) {
            $conversation = Conversation::findOrFail($conversationId);
        } else {
            $listing = Listing::findOrFail($listingId);
            $conversation = Conversation::firstOrCreate([
                'listing_id' => $listing->id,
                'buyer_id' => $user->id,
                'business_id' => $listing->business_id,
            ]);
        }

        if (!$this->canAccessConversation($user, $conversation)) {
            return response()->json(['error' => 'Unauthorized'], 403);
        }

        $attachmentPath = null;
        $type = 'text';

        if ($request->hasFile('attachment')) {
            $file = $request->file('attachment');
            $filename = time() . '_' . uniqid() . '.' . $file->getClientOriginalExtension();
            $attachmentPath = $file->storeAs('chat/attachments', $filename, 'public');
            $type = 'image';
        }

        if (!$attachmentPath && empty($request->message)) {
             return response()->json(['error' => 'Message or attachment required'], 422);
        }

        $message = Message::create([
            'conversation_id' => $conversation->id,
            'sender_id' => $user->id,
            'message' => $request->message ?? '',
            'attachment' => $attachmentPath,
            'type' => $type,
        ]);

        $conversation->update(['last_message_at' => now()]);

        // Trigger Notifications (Reuse Logic from API Controller, or service)
        // ideally extracting to a Service is better, but for now copying logic is acceptable for speed
        $this->sendNotifications($user, $conversation, $message);

        return response()->json([
            'success' => true,
            'message' => [
                'id' => $message->id,
                'message' => $message->message,
                'attachment' => $message->attachment ? asset('storage/' . $message->attachment) : null,
                'type' => $message->type,
                'is_me' => true,
                'sender_name' => $user->full_name ?? $user->username,
                'sender_avatar' => $user->profile_photo_url,
                'created_at' => $message->created_at->format('g:i A'),
                'created_at_raw' => $message->created_at->toIso8601String(),
                'read_at' => null,
            ],
            'conversation_id' => $conversation->id, // Return ID in case it was created new
        ]);
    }

    /**
     * Mark conversation as read (manual trigger).
     */
    public function markAsRead(Conversation $conversation)
    {
        $user = auth('web')->user();

        if (!$this->canAccessConversation($user, $conversation)) {
            return response()->json(['error' => 'Unauthorized'], 403);
        }

        $conversation->messages()
            ->where('sender_id', '!=', $user->id)
            ->whereNull('read_at')
            ->update(['read_at' => now()]);

        return response()->json(['success' => true]);
    }

    /**
     * Check access permission.
     */
    private function canAccessConversation($user, Conversation $conversation): bool
    {
        // Is buyer?
        if ($conversation->buyer_id == $user->id) {
            return true;
        }

        // Is business member/owner?
        // We can check owned businesses efficiently
        $isMember = Business::where('id', $conversation->business_id)
            ->where(function ($q) use ($user) {
                // Check if owner
                $q->where('owner_user_id', $user->id)
                // Check if member
                  ->orWhereHas('memberships', function ($mq) use ($user) {
                      $mq->where('user_id', $user->id);
                  });
            })
            ->exists();

        return $isMember;
    }

    /**
     * Send notifications (Email/Push/DB).
     */
    private function sendNotifications($user, $conversation, $message)
    {
        try {
            $fcmService = resolve(\App\Services\FcmService::class);
            $senderName = $user->full_name ?? $user->username ?? 'Someone';
            $title = "You have a new message from {$senderName}";
            $body = ($message->type === 'image') ? 'Sent an image' : ($message->message ?? 'New message');
            
            $notificationData = [
                'type' => 'chat',
                'conversation_id' => (string) $conversation->id,
                'sender_id' => (string) $user->id,
                'listing_id' => (string) $conversation->listing_id,
            ];

            if ($user->id == $conversation->buyer_id) {
                // Buyer sent message to business
                
                // 1. Push
                $fcmService->sendToBusiness($conversation->business_id, $title, $body, $notificationData);

                // 2. Email/DB to all business members
                $business = Business::with(['memberships.user', 'owner'])->find($conversation->business_id);
                if ($business) {
                    $recipients = collect();
                    if ($business->owner) $recipients->push($business->owner);
                    foreach ($business->memberships as $membership) {
                        if ($membership->user) $recipients->push($membership->user);
                    }
                    
                    $uniqueRecipients = $recipients->unique('id');

                    foreach ($uniqueRecipients as $recipient) {
                        if ($recipient->id != $user->id) { 
                             if ($recipient->email) {
                                \Mail::to($recipient->email)->queue(
                                    new \App\Mail\NewMessageNotification(
                                        $message, $user, 
                                        $recipient->full_name ?? $recipient->username ?? 'Business Manager'
                                    )
                                );
                            }
                            $recipient->notify(new \App\Notifications\GeneralNotification($title, $body, $notificationData));
                        }
                    }
                }

            } else {
                // Seller sent message to buyer
                
                // 1. Push
                $fcmService->sendToUser($conversation->buyer_id, $title, $body, $notificationData);

                // 2. Email/DB
                $buyer = \App\Models\User::find($conversation->buyer_id);
                if ($buyer) {
                    if ($buyer->email) {
                        \Mail::to($buyer->email)->queue(
                            new \App\Mail\NewMessageNotification(
                                $message, $user, 
                                $buyer->full_name ?? $buyer->username ?? 'User'
                            )
                        );
                    }
                    $buyer->notify(new \App\Notifications\GeneralNotification($title, $body, $notificationData));
                }
            }
        } catch (\Exception $e) {
            \Log::error('Failed to send chat notifications: ' . $e->getMessage());
        }
    }
}
