<?php

namespace App\Http\Controllers\Web;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;

use App\Models\User;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\ValidationException;

use App\Models\LoginSession;
use App\Mail\LoginNoticeEmail;
use Illuminate\Support\Facades\Mail;
use Carbon\Carbon;

class AuthController extends Controller
{
    public function showLogin()
    {
        return view('web.auth.login');
    }

    public function login(Request $request)
    {
        $credentials = $request->validate([
            'login' => 'required|string',
            'password' => 'required|string',
        ]);

        // Support both email and phone/username login like the API
        $field = filter_var($credentials['login'], FILTER_VALIDATE_EMAIL) ? 'email' : 'phone';
        
        // Set session lifetime based on "Remember Me"
        if ($request->filled('remember')) {
            config(['session.lifetime' => 43200]); // 30 days
        } else {
            config(['session.lifetime' => 2880]); // 48 hours
        }

        if (Auth::guard('web')->attempt([$field => $credentials['login'], 'password' => $credentials['password']], $request->filled('remember'))) {
            $request->session()->regenerate();
            $user = Auth::guard('web')->user();
            if ($user) {
                $user->recordSuccessfulLogin($request->ip());
                $this->handleSuccessfulLogin($request, $user);
            }
            return redirect()->intended(route('web.home'));
        }

        // Check for legacy password if standard login fails
        $user = User::where($field, $credentials['login'])->first();
        if ($user && $user->v1_password_hash && $user->v1_salt) {
            $concatenatedPassword = $this->concatPasswordWithSalt($credentials['password'], $user->v1_salt);
            
            if (password_verify($concatenatedPassword, $user->v1_password_hash)) {
                // Legacy success! Upgrade the password
                $user->password = $credentials['password'];
                $user->v1_password_hash = null;
                $user->v1_salt = null;
                $user->save();

                // Set session lifetime based on "Remember Me"
                if ($request->filled('remember')) {
                    config(['session.lifetime' => 43200]); // 30 days
                } else {
                    config(['session.lifetime' => 2880]); // 48 hours
                }

                // Log them in
                Auth::guard('web')->login($user, $request->filled('remember'));
                $request->session()->regenerate();
                
                if ($user) {
                    $user->recordSuccessfulLogin($request->ip());
                    $this->handleSuccessfulLogin($request, $user);
                }
                return redirect()->intended(route('web.home'));
            }
        }

        throw ValidationException::withMessages([
            'login' => [trans('auth.failed')],
        ]);
    }

    /**
     * Handle post-login actions (Session tracking, Email, key updates).
     */
    private function handleSuccessfulLogin(Request $request, User $user)
    {
        // 1. Create LoginSession
        $session = null;
        try {
            $session = LoginSession::create([
                'user_id' => $user->id,
                'token_id' => $request->session()->getId(), // Use Laravel Session ID for Web
                'ip_address' => $request->ip(),
                'user_agent' => $request->userAgent(),
                'issued_at' => now(),
                'expires_at' => now()->addMinutes(config('session.lifetime')),
                'logged_in_at' => now(),
            ]);
        } catch (\Throwable $e) {
            \Log::warning('Failed to save login session (web)', [
                'user_id' => $user->id,
                'error' => $e->getMessage(),
            ]);
        }

        // 2. Send Email
        if ($session) {
            try {
                Mail::to($user->email)->send(new LoginNoticeEmail($user, $session));
            } catch (\Throwable $e) {
                \Log::warning('Failed to send login notice email (web)', [
                    'user_id' => $user->id,
                    'email' => $user->email,
                    'error' => $e->getMessage(),
                ]);
            }
        }
    }

    private function concatPasswordWithSalt(string $password, string $salt): string
    {
        $random_salt_length = 32; 
        
        if ($random_salt_length % 2 == 0) {
            $mid = $random_salt_length / 2;
        } else {
            $mid = ($random_salt_length - 1) / 2;
        }

        return substr($salt, 0, $mid - 1) . $password . substr($salt, $mid, $random_salt_length - 1);
    }

    public function showRegister()
    {
        return view('web.auth.register');
    }

    public function register(Request $request)
    {
        // Normalize phone if present
        if ($request->filled('phone')) {
            $phone = preg_replace('/[^0-9]/', '', $request->phone);
            // If it starts with 256, it's already got the country code
            if (str_starts_with($phone, '256')) {
                $phone = '+' . $phone;
            } else {
                // Otherwise trim 0 and add +256
                $phone = '+256' . ltrim($phone, '0');
            }
            $request->merge(['phone' => $phone]);
        }

        $request->validate([
            'full_name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users',
            'phone' => 'nullable|string|max:15|unique:users,phone',
            'password' => 'required|string|min:8|confirmed',
            'referral_code' => 'nullable|string|exists:users,referral_code',
        ]);

        // Find referrer if code is provided
        $referredBy = null;
        if ($request->referral_code) {
            $referrer = User::where('referral_code', $request->referral_code)->first();
            if ($referrer) {
                $referredBy = $referrer->id;
                $referrer->increment('referral_count');
            }
        }

        // Create the user
        $user = User::create([
            'full_name' => $request->full_name,
            'email' => $request->email,
            'phone' => $request->phone, // Already normalized above
            'password' => $request->password, // Will be hashed automatically by 'hashed' cast
            'referred_by' => $referredBy,
            'registration_source' => 'web',
            'registration_ip' => $request->ip(),
            'status' => User::STATUS_ACTIVE,
            'role' => User::ROLE_USER,
            'country_code' => 'UG',
            'country' => 'Uganda',
        ]);

        // Set default session lifetime (48 hours)
        config(['session.lifetime' => 2880]);

        Auth::guard('web')->login($user);
        $request->session()->regenerate();
        
        if ($user) {
            $user->recordSuccessfulLogin($request->ip());
            $this->handleSuccessfulLogin($request, $user);
        }

        return redirect()->route('web.home')->with('success', 'Welcome to Yilick! Your account has been created.');
    }

    public function logout(Request $request)
    {
        Auth::guard('web')->logout();

        $request->session()->invalidate();
        $request->session()->regenerateToken();

        return redirect('/');
    }

    public function showForgotPassword()
    {
        return view('web.auth.forgot-password');
    }

    public function forgotPassword(Request $request)
    {
        $request->validate(['email' => 'required|email']);

        $status = \Illuminate\Support\Facades\Password::broker()->sendResetLink(
            $request->only('email')
        );

        return $status === \Illuminate\Support\Facades\Password::RESET_LINK_SENT
            ? back()->with('success', __($status))
            : back()->withErrors(['email' => __($status)]);
    }
}
