<?php

namespace App\Http\Controllers\Api\V1;

use App\Http\Controllers\Controller;
use App\Http\Resources\ApiResponse;
use App\Models\Business;
use App\Models\BusinessBlock;
use App\Models\Listing;
use App\Models\ListingReport;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class UserContentController extends Controller
{
    public function reportListing(Request $request, Listing $listing): JsonResponse
    {
        $validated = $request->validate([
            'reason' => 'required|string|max:255',
            'explanation' => 'nullable|string|max:1000',
        ]);

        $userId = auth()->id();
        $ip = $request->ip();
        $hasIpColumn = \Illuminate\Support\Facades\Schema::hasColumn('listing_reports', 'ip_address');

        // If authenticated, we prevent duplicate reports. 
        // If guest, we allow the report (though we could rate limit by IP).
        if ($userId) {
            $data = [
                'reason' => $validated['reason'],
                'explanation' => $validated['explanation'] ?? null,
            ];
            if ($hasIpColumn) {
                $data['ip_address'] = $ip;
            }

            ListingReport::firstOrCreate(
                [
                    'listing_id' => $listing->id,
                    'user_id' => $userId,
                ],
                $data
            );
        } else {
            // For guests, we just create the report.
            $data = [
                'listing_id' => $listing->id,
                'user_id' => null,
                'reason' => $validated['reason'],
                'explanation' => $validated['explanation'] ?? null,
            ];
            if ($hasIpColumn) {
                $data['ip_address'] = $ip;
            }

            ListingReport::create($data);
        }

        return ApiResponse::success(null, 'Listing reported successfully. We will review it shortly.');
    }

    public function blockBusiness(Request $request, Business $business): JsonResponse
    {
        BusinessBlock::firstOrCreate([
            'business_id' => $business->id,
            'user_id' => auth()->id(),
        ]);

        return ApiResponse::success(null, 'Business blocked. You will no longer see their listings.');
    }
}
