<?php

namespace App\Http\Controllers\Api\V1;

use App\Http\Controllers\Controller;
use App\Models\Business;
use App\Models\PosShift;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class PosShiftController extends Controller
{
    /**
     * Get current open shift for the user.
     */
    public function current(Request $request, Business $business)
    {
        if (!$business->isMember($request->user())) {
             return response()->json(['message' => 'Unauthorized'], 403);
        }

        $shift = PosShift::where('business_id', $business->id)
            ->where('user_id', $request->user()->id)
            ->where('status', 'open')
            ->first();

        if (!$shift) {
            return response()->json(['shift' => null]);
        }

        return $this->show($request, $business, $shift);
    }

    /**
     * Open a new shift (Cash Drawer Start).
     */
    public function open(Request $request, Business $business)
    {
        if (!$business->isMember($request->user())) {
             return response()->json(['message' => 'Unauthorized'], 403);
        }

        $validator = Validator::make($request->all(), [
            'opening_balance' => 'required|numeric',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        // Close any existing open shifts for THIS user
        PosShift::where('business_id', $business->id)
            ->where('user_id', $request->user()->id)
            ->where('status', 'open')
            ->update([
                'status' => 'closed',
                'closed_at' => now(),
            ]);

        $shift = PosShift::create([
            'business_id' => $business->id,
            'user_id' => $request->user()->id,
            'opening_balance' => $request->opening_balance,
            'status' => 'open',
            'opened_at' => now(),
        ]);

        return response()->json(['shift' => $shift, 'status' => 'success']);
    }

    /**
     * Close a shift (Cash Drawer End).
     */
    public function close(Request $request, Business $business, PosShift $shift)
    {
        if (!$business->isMember($request->user()) || $shift->business_id != $business->id) {
             return response()->json(['message' => 'Unauthorized'], 403);
        }

        $validator = Validator::make($request->all(), [
            'actual_closing_balance' => 'required|numeric',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        // Calculate expected closing balance based on cash sales
        $totalCashSales = \App\Models\PosPayment::whereHas('order', function($q) use ($business, $shift) {
            $q->where('business_id', $business->id)
              ->where('cashier_user_id', $shift->user_id)
              ->where('created_at', '>=', $shift->opened_at);
        })->where('payment_method', 'cash')->sum('amount');

        $expectedClosing = $shift->opening_balance + $totalCashSales;

        $shift->update([
            'closing_balance' => $expectedClosing,
            'actual_closing_balance' => $request->actual_closing_balance,
            'status' => 'closed',
            'closed_at' => now(),
        ]);

        return response()->json([
            'shift' => $shift,
            'discrepancy' => $request->actual_closing_balance - $expectedClosing,
            'status' => 'success'
        ]);
    }

    /**
     * List past shifts for the business.
     */
    public function index(Request $request, Business $business)
    {
        if (!$business->isMember($request->user())) {
             return response()->json(['message' => 'Unauthorized'], 403);
        }

        $query = PosShift::where('business_id', $business->id)
            ->select('pos_shifts.*')
            ->addSelect(['total_sales' => \App\Models\PosPayment::whereHas('order', function($q) use ($business) {
                $q->where('business_id', $business->id)
                  ->whereColumn('cashier_user_id', 'pos_shifts.user_id')
                  ->whereColumn('created_at', '>=', 'pos_shifts.opened_at')
                  ->where(function($q) {
                      $q->whereColumn('created_at', '<=', 'pos_shifts.closed_at')
                        ->orWhereNull('pos_shifts.closed_at');
                  });
            })->selectRaw('COALESCE(SUM(amount), 0)')])
            ->addSelect(['total_expenses' => \App\Models\PosExpense::whereColumn('pos_shift_id', 'pos_shifts.id')
                ->where('payment_method', 'cash')
                ->selectRaw('COALESCE(SUM(amount), 0)')])
            ->with(['user:id,full_name']);

        // Filter by Status
        if ($request->has('status')) {
            $query->where('status', $request->status);
        }

        // Filter by User
        if ($request->has('user_id')) {
            $query->where('user_id', $request->user_id);
        }

        // Filter by Date Range
        if ($request->has('start_date')) {
            $query->whereDate('opened_at', '>=', $request->start_date);
        }
        if ($request->has('end_date')) {
            $query->whereDate('opened_at', '<=', $request->end_date);
        }

        $shifts = $query->orderBy('created_at', 'desc')
            ->cursorPaginate(20);

        return response()->json($shifts);
    }

    /**
     * Get detailed analytics for a shift.
     */
    public function show(Request $request, Business $business, PosShift $shift)
    {
        if (!$business->isMember($request->user()) || $shift->business_id != $business->id) {
             return response()->json(['message' => 'Unauthorized'], 403);
        }

        // Load relationships
        $shift->load(['user:id,full_name']);

        // Calculate Sales Breakdown
        // We define the time range as opened_at -> closed_at (or now if open)
        $endTime = $shift->closed_at ?? now();

        $payments = \App\Models\PosPayment::whereHas('order', function($q) use ($business, $shift, $endTime) {
            $q->where('business_id', $business->id)
              ->where('cashier_user_id', $shift->user_id)
              ->whereBetween('created_at', [$shift->opened_at, $endTime]);
        })->selectRaw('payment_method, SUM(amount) as total')
          ->groupBy('payment_method')
          ->get();

        $salesBreakdown = $payments->mapWithKeys(function ($item) {
            return [$item->payment_method => (float)$item->total];
        });

        $totalSales = $salesBreakdown->sum();
        $cashSales = $salesBreakdown['cash'] ?? 0;
        
        // Calculate Cash Expenses for this shift
        $expenses = \App\Models\PosExpense::where('pos_shift_id', $shift->id)
            ->with(['user:id,full_name'])
            ->orderBy('expense_date', 'desc')
            ->get();
            
        $cashExpenses = $expenses->where('payment_method', 'cash')->sum('amount');

        // Expected Cash in Drawer = Opening Float + Cash Sales - Cash Expenses
        $expectedCash = $shift->opening_balance + $cashSales - $cashExpenses;
        
        // Discrepancy (if closed)
        $discrepancy = null;
        if ($shift->status === 'closed' && $shift->actual_closing_balance !== null) {
            $discrepancy = $shift->actual_closing_balance - $expectedCash;
        }

        return response()->json([
            'shift' => $shift,
            'analytics' => [
                'total_sales' => $totalSales,
                'cash_sales' => $cashSales,
                'cash_expenses' => (float)$cashExpenses,
                'expenses' => $expenses,
                'expected_cash' => (float)$expectedCash,
                'actual_cash' => $shift->actual_closing_balance,
                'discrepancy' => (float)$discrepancy,
                'sales_by_method' => $salesBreakdown,
            ]
        ]);
    }
}
