<?php

namespace App\Http\Controllers\Api\V1;

use App\Http\Controllers\Controller;
use App\Models\Business;
use App\Models\PosOrder;
use App\Models\PosRefund;
use App\Models\Listing;
use App\Models\PosCustomerLedger;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;

class PosRefundController extends Controller
{
    /**
     * Process a refund/return.
     */
    public function store(Request $request, Business $business)
    {
        if (!$business->isMember($request->user())) {
             return response()->json(['message' => 'Unauthorized'], 403);
        }

        $validator = Validator::make($request->all(), [
            'pos_order_id' => 'required|exists:pos_orders,id',
            'amount' => 'required|numeric|min:0',
            'items' => 'required|array', // [{id, qty, price}]
            'reason' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        $order = PosOrder::where('id', $request->pos_order_id)
            ->where('business_id', $business->id)
            ->firstOrFail();

        if ($request->amount > ($order->total_amount - $order->refunded_amount)) {
            return response()->json(['message' => 'Refund amount exceeds remaining balance'], 422);
        }

        DB::beginTransaction();
        try {
            // Create Refund Record
            $refund = PosRefund::create([
                'business_id' => $business->id,
                'pos_order_id' => $order->id,
                'user_id' => $request->user()->id,
                'amount' => $request->amount,
                'items' => $request->items,
                'reason' => $request->reason,
            ]);

            // Update Order Status
            $order->increment('refunded_amount', $request->amount);
            $order->status = ($order->refunded_amount >= $order->total_amount) ? 'refunded' : 'partially_refunded';
            $order->save();

            // Restore Stock
            foreach ($request->items as $item) {
                $listing = Listing::find($item['id']);
                if ($listing) {
                    $listing->incrementStock(
                        $item['qty'], 
                        'return', 
                        'REFUND-'.$refund->id, 
                        $request->user()->id
                    );
                }
            }

            // Adjust Customer Debt if applicable
            if ($order->pos_customer_id) {
                $customer = $order->customer;
                // If the order had credit payments, reduce the debt
                $creditPaymentAmount = $order->payments()->where('payment_method', 'credit')->sum('amount');
                if ($creditPaymentAmount > 0) {
                    // Logic: We reduce the debt by the refund amount, but not more than the original debt
                    $reduction = min($request->amount, $customer->balance);
                    $customer->decrement('balance', $reduction);

                    PosCustomerLedger::create([
                        'pos_customer_id' => $customer->id,
                        'business_id' => $business->id,
                        'pos_order_id' => $order->id,
                        'type' => 'credit', // Credit means payment/reduction in debt
                        'amount' => $reduction,
                        'notes' => 'Debt reduced due to refund - Refund #' . $refund->id,
                    ]);
                }
            }

            DB::commit();
            return response()->json(['refund' => $refund, 'status' => 'success']);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json(['message' => 'Refund failed', 'error' => $e->getMessage()], 500);
        }
    }
}
