<?php

namespace App\Http\Controllers\Api\V1;

use App\Http\Controllers\Controller;
use App\Models\Business;
use App\Models\PosExpense;
use App\Models\PosShift;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class PosExpenseController extends Controller
{
    /**
     * List expenses for the business.
     */
    public function index(Request $request, Business $business)
    {
        if (!$business->isMember($request->user())) {
            return response()->json(['message' => 'Unauthorized'], 403);
        }

        $query = PosExpense::where('business_id', $business->id)
            ->with(['user:id,full_name', 'shift:id']);

        // Filter by Category
        if ($request->has('category')) {
            $query->where('category', $request->category);
        }

        // Filter by Payment Method
        if ($request->has('payment_method')) {
            $query->where('payment_method', $request->payment_method);
        }

        // Filter by Date Range
        if ($request->has('start_date')) {
            $query->whereDate('expense_date', '>=', $request->start_date);
        }
        if ($request->has('end_date')) {
            $query->whereDate('expense_date', '<=', $request->end_date);
        }

        // Filter by Shift ID
        if ($request->has('pos_shift_id')) {
            $query->where('pos_shift_id', $request->pos_shift_id);
        }

        $expenses = $query->orderBy('expense_date', 'desc')
            ->orderBy('created_at', 'desc')
            ->paginate(30);

        return response()->json($expenses);
    }

    /**
     * Store a new business expense.
     */
    public function store(Request $request, Business $business)
    {
        if (!$business->isMember($request->user())) {
            return response()->json(['message' => 'Unauthorized'], 403);
        }

        $validator = Validator::make($request->all(), [
            'amount' => 'required|numeric|min:0',
            'reason' => 'required|string|max:255',
            'category' => 'required|string',
            'payment_method' => 'required|string|in:cash,mobile_money,bank,card',
            'expense_date' => 'sometimes|date',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        $shiftId = null;
        
        // If payment method is cash, automatically link to the user's current open shift if available
        if ($request->payment_method === 'cash') {
            $currentShift = PosShift::where('business_id', $business->id)
                ->where('user_id', $request->user()->id)
                ->where('status', 'open')
                ->first();
            
            if ($currentShift) {
                $shiftId = $currentShift->id;
            }
        }

        $expense = PosExpense::create([
            'business_id' => $business->id,
            'pos_shift_id' => $shiftId,
            'user_id' => $request->user()->id,
            'amount' => $request->amount,
            'reason' => $request->reason,
            'category' => $request->category,
            'payment_method' => $request->payment_method,
            'expense_date' => $request->expense_date ?? now(),
        ]);

        return response()->json(['expense' => $expense, 'status' => 'success']);
    }

    /**
     * Delete an expense.
     */
    public function destroy(Request $request, Business $business, PosExpense $expense)
    {
        if (!$business->isMember($request->user()) || $expense->business_id != $business->id) {
            return response()->json(['message' => 'Unauthorized'], 403);
        }

        $expense->delete();

        return response()->json(['status' => 'success']);
    }
}
