<?php

namespace App\Http\Controllers\Api\V1;

use App\Http\Controllers\Controller;
use App\Models\Business;
use App\Models\Listing;
use App\Models\PosOrder;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;

use App\Services\InventoryService;

class PosController extends Controller
{
    protected $inventoryService;

    public function __construct(InventoryService $inventoryService)
    {
        $this->inventoryService = $inventoryService;
    }
    /**
     * Pull updates (Down-sync).
     * Client sends last_sync_timestamp.
     * Sever returns items updated since then.
     */
    public function pull(Request $request, Business $business)
    {
        // Authorize: User must be member of business
        if (!$business->isMember($request->user())) {
             return response()->json(['message' => 'Unauthorized'], 403);
        }

        $lastSync = $request->input('last_sync_timestamp');
        $date = $lastSync ? Carbon::createFromTimestamp($lastSync) : Carbon::create(2000, 1, 1);

        // Fetch updated listings
        $updatedListings = Listing::where('business_id', $business->id)
            ->where(function($q) use ($date) {
                $q->where('updated_at', '>', $date)
                  ->orWhere('created_at', '>', $date);
            })
            ->withTrashed() // Include deleted items to tell POS to remove them
            ->with(['media']) // Eager load media to prevent N+1
            ->orderBy('id') // Required for cursor pagination
            // Only fetch fields relevant to POS to save bandwidth
            ->cursorPaginate(500, [
                'id', 'business_id', 'title', 'price_amount', 'purchase_price_amount', 'quantity', 'store_quantity',
                'sku', 'barcode', 'stock_alert_level', 'pos_quick_access', 
                'status', 'deleted_at', 'updated_at', 'category'
            ]);

        // Transform for POS
        $payload = collect($updatedListings->items())->map(function ($item) {
            return [
                'id' => $item->id,
                'title' => $item->title,
                'price' => $item->price_amount,
                'stock' => $item->quantity,
                'store_stock' => $item->store_quantity ?? 0, // Warehouse stock
                'sku' => $item->sku,
                'barcode' => $item->barcode,
                'is_deleted' => !is_null($item->deleted_at),
                'quick_access' => $item->pos_quick_access,
                'updated_at' => $item->updated_at->timestamp,
                'category' => $item->category,
                'thumbnail_url' => $item->thumbnail_url,
                'cost_price' => $item->purchase_price_amount,
            ];
        });

        return response()->json([
            'updates' => $payload,
            'next_cursor' => $updatedListings->nextCursor()?->encode(),
            'timestamp' => now()->timestamp,
        ]);
    }

    /**
     * Push actions (Up-sync).
     * Client sends offline sales/orders.
     */
    public function push(Request $request, Business $business)
    {
        if (!$business->isMember($request->user())) {
             return response()->json(['message' => 'Unauthorized'], 403);
        }

        $validator = Validator::make($request->all(), [
            'orders' => 'required|array',
            'orders.*.offline_uuid' => 'required|string',
            'orders.*.total_amount' => 'required|numeric',
            'orders.*.items' => 'required|array',
            'orders.*.payments' => 'required|array', // New requirement for split payments
            'orders.*.timestamp' => 'numeric',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        $orders = $request->input('orders');
        $processed = [];

        DB::beginTransaction();
        try {
            foreach ($orders as $orderData) {
                if (\App\Models\PosOrder::where('offline_uuid', $orderData['offline_uuid'])->exists()) {
                    $processed[] = $orderData['offline_uuid'];
                    continue;
                }

                // Ensure customer ID is integer if provided
                $customerId = isset($orderData['pos_customer_id']) ? (int)$orderData['pos_customer_id'] : null;

                $posOrder = \App\Models\PosOrder::create([
                    'business_id' => $business->id,
                    'cashier_user_id' => $request->user()->id,
                    'pos_customer_id' => $customerId,
                    'total_amount' => (float)$orderData['total_amount'],
                    'tax_amount' => (float)($orderData['tax_amount'] ?? 0),
                    'discount_amount' => (float)($orderData['discount_amount'] ?? 0),
                    'items' => $orderData['items'],
                    'payment_method' => 'split',
                    'receipt_number' => $orderData['receipt_number'] ?? null,
                    'offline_uuid' => $orderData['offline_uuid'],
                    'synced_at' => now(),
                    'status' => $orderData['status'] ?? 'completed',
                ]);

                // Manually set created_at from timestamp if provided
                if (isset($orderData['timestamp']) && $orderData['timestamp'] > 0) {
                    try {
                        $posOrder->created_at = \Carbon\Carbon::createFromTimestamp((int)$orderData['timestamp']);
                        $posOrder->save();
                    } catch (\Exception $e) {
                         // Fallback to now if timestamp is corrupted
                    }
                }

                // Record Payments
                if (isset($orderData['payments']) && is_array($orderData['payments'])) {
                    foreach ($orderData['payments'] as $paymentData) {
                        $posOrder->payments()->create([
                            'payment_method' => $paymentData['method'],
                            'amount' => (float)$paymentData['amount'],
                            'reference_id' => $paymentData['reference'] ?? null,
                        ]);

                        // Handle Debt (Credit Sales)
                        if ($paymentData['method'] === 'credit' && $posOrder->pos_customer_id) {
                            $customer = $posOrder->customer;
                            if ($customer) {
                                $customer->increment('balance', (float)$paymentData['amount']);
                                \App\Models\PosCustomerLedger::create([
                                    'pos_customer_id' => $customer->id,
                                    'business_id' => $business->id,
                                    'pos_order_id' => $posOrder->id,
                                    'type' => 'debit',
                                    'amount' => (float)$paymentData['amount'],
                                    'notes' => 'Purchase on credit - Order #' . ($posOrder->receipt_number ?? $posOrder->id),
                                ]);
                            }
                        }
                    }
                }

                // Update Loyalty Points
                if ($posOrder->pos_customer_id) {
                    $customer = $posOrder->customer;
                    if ($customer) {
                        $customer->increment('loyalty_points', floor($posOrder->total_amount / 1000));
                    }
                }

                // Update Stock (Using FIFO Service)
                foreach ($orderData['items'] as $item) {
                    if (!isset($item['id']) || !isset($item['qty'])) continue;

                    $listing = \App\Models\Listing::find($item['id']);
                    if ($listing) {
                        try {
                            $this->inventoryService->deductShopStock(
                                $listing,
                                (int)$item['qty'],
                                'sale_pos',
                                'POS-'.$posOrder->id,
                                $request->user()->id
                            );
                        } catch (\Exception $e) {
                            // If FIFO deduction fails (e.g. no batches), fallback to legacy decrement
                            // to ensure the sale is recorded even if inventory tracking is messy
                            $listing->decrementStock((int)$item['qty'], 'sale_pos_legacy', 'POS-'.$posOrder->id, $request->user()->id);
                        }
                    }
                }
                
                $processed[] = $orderData['offline_uuid'];
            }
            DB::commit();

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json(['message' => 'Sync failed', 'error' => $e->getMessage()], 500);
        }

        return response()->json([
            'processed_uuids' => $processed,
            'status' => 'success'
        ]);
    }

    /**
     * Bulk update SKUs for listings.
     */
    public function bulkUpdateSkus(Request $request, Business $business)
    {
        if (!$business->isMember($request->user())) {
             return response()->json(['message' => 'Unauthorized'], 403);
        }

        $validator = Validator::make($request->all(), [
            'updates' => 'required|array',
            'updates.*.listing_id' => 'required|integer',
            'updates.*.sku' => 'required|string',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        $updates = $request->input('updates');
        $processed = [];
        $errors = [];

        DB::beginTransaction();
        try {
            foreach ($updates as $update) {
                $listing = Listing::where('business_id', $business->id)
                    ->where('id', $update['listing_id'])
                    ->first();

                if (!$listing) {
                    $errors[] = "Listing ID {$update['listing_id']} not found for this business.";
                    continue;
                }

                // Check for SKU uniqueness WITHIN the business
                $exists = Listing::where('business_id', $business->id)
                    ->where('sku', $update['sku'])
                    ->where('id', '!=', $listing->id)
                    ->exists();

                if ($exists) {
                    $errors[] = "SKU '{$update['sku']}' already taken for item: {$listing->title}. Skipping.";
                    continue;
                }

                $listing->update([
                    'sku' => $update['sku'],
                    'barcode' => $update['sku'], // Sync barcode with SKU for POS
                ]);

                $processed[] = $listing->id;
            }
            DB::commit();
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json(['message' => 'Bulk update failed', 'error' => $e->getMessage()], 500);
        }

        return response()->json([
            'processed_count' => count($processed),
            'processed_ids' => $processed,
            'errors' => $errors,
            'status' => 'success'
        ]);
    }

    /**
     * List orders for the business.
     */
    public function indexOrders(Request $request, Business $business)
    {
        if (!$business->isMember($request->user())) {
             return response()->json(['message' => 'Unauthorized'], 403);
        }

        $query = PosOrder::where('business_id', $business->id)
            ->with(['customer:id,name', 'cashier:id,full_name', 'payments']);

        // Filter by Date Range
        if ($request->has('start_date')) {
            $query->whereDate('created_at', '>=', $request->start_date);
        }
        if ($request->has('end_date')) {
            $query->whereDate('created_at', '<=', $request->end_date);
        }
        
        // Filter by Status
        if ($request->has('status') && $request->status !== 'all') {
            $query->where('status', $request->status);
        }

        // Search by Recipe Number or Customer Name
        if ($request->has('query') && !empty($request->query('query'))) {
            $searchTerm = $request->query('query');
            $query->where(function($q) use ($searchTerm) {
                $q->where('receipt_number', 'LIKE', "%$searchTerm%")
                  ->orWhere('offline_uuid', 'LIKE', "%$searchTerm%")
                  ->orWhereHas('customer', function($cq) use ($searchTerm) {
                      $cq->where('name', 'LIKE', "%$searchTerm%");
                  });
            });
        }

        $orders = $query->orderBy('created_at', 'desc')
            ->paginate(30);

        return response()->json($orders);
    }

    /**
     * Get details of a specific order.
     */
    public function showOrder(Request $request, Business $business, PosOrder $order)
    {
        if (!$business->isMember($request->user())) {
             return response()->json(['message' => 'Unauthorized'], 403);
        }

        if ($order->business_id != $business->id) {
            return response()->json(['message' => 'Order not found for this business'], 404);
        }

        $order->load(['payments', 'customer', 'cashier', 'business']);

        // Determine human-readable payment summary
        $paymentMethods = $order->payments->pluck('payment_method')->unique()->map(function($m) {
            return ucfirst(str_replace('_', ' ', $m));
        });
        
        if ($paymentMethods->isEmpty()) {
            $order->primary_payment_method = 'Unpaid';
        } else {
            $order->primary_payment_method = $paymentMethods->implode(' & ');
        }

        // Fetch Inventory logs to get batch info and accurate profit
        $fifoLogs = \App\Models\InventoryLog::where('reference_id', 'POS-' . $order->id)
            ->with(['batch:id,cost_price,received_at,reference_note', 'listing:id,title,purchase_price_amount'])
            ->get();

        // Group by listing to calculate cost and gather batch info
        $itemCosts = [];
        foreach ($fifoLogs as $log) {
            $listingId = $log->listing_id;
            
            // If log has a batch, use batch cost. Otherwise, fallback to listing purchase price.
            $batchCost = $log->batch->cost_price ?? $log->listing->purchase_price_amount ?? 0;
            $totalLogCost = abs($log->change_amount) * $batchCost;
            
            if (!isset($itemCosts[$listingId])) {
                $itemCosts[$listingId] = [
                    'cost' => 0,
                    'batches' => [],
                    'fallback_used' => false
                ];
            }
            
            $itemCosts[$listingId]['cost'] += $totalLogCost;
            
            if ($log->batch) {
                // Determine a friendly name for the batch
                $batchId = $log->batch->reference_note ?? $log->batch->id;
                
                $itemCosts[$listingId]['batches'][] = [
                    'batch_number' => $batchId,
                    'quantity' => abs($log->change_amount),
                    'cost_price' => (float)$log->batch->cost_price,
                ];
            } else {
                $itemCosts[$listingId]['fallback_used'] = true;
            }
        }

        // Efficiently enrich order items with listing data (thumbnails)
        $items = $order->items;
        foreach ($items as &$item) {
            $listingId = $item['id'] ?? null;
            if ($listingId) {
                // Find matching log to get listing data more efficiently
                $log = $fifoLogs->firstWhere('listing_id', $listingId);
                if ($log && $log->listing) {
                    $item['thumbnail_url'] = $log->listing->thumbnail_url;
                } else {
                    // Fallback for items that might not have logs (unlikely but safe)
                    $listing = \App\Models\Listing::find($listingId);
                    if ($listing) {
                        $item['thumbnail_url'] = $listing->thumbnail_url;
                    }
                }
            }
        }
        $order->items = $items;

        return response()->json([
            'order' => $order,
            'fifo_details' => $itemCosts,
            'status' => 'success'
        ]);
    }

    /**
     * Generate an AI summary for the order.
     */
    public function summarizeOrder(Request $request, Business $business, PosOrder $order, \App\Services\GeminiService $geminiService)
    {
        if (!$business->isMember($request->user())) {
            return response()->json(['message' => 'Unauthorized'], 403);
        }

        if ($order->business_id != $business->id) {
            return response()->json(['message' => 'Order not found for this business'], 404);
        }

        $order->load(['customer']);

        // 1. Check if summary already exists (Cost Optimization)
        if (!empty($order->ai_summary)) {
            return response()->json([
                'summary' => $order->ai_summary,
                'status' => 'success',
                'source' => 'cache'
            ]);
        }

        // Fetch Inventory logs to get batch info and accurate profit for AI analysis
        $fifoLogs = \App\Models\InventoryLog::where('reference_id', 'POS-' . $order->id)
            ->with(['batch:id,cost_price,received_at', 'listing:id,purchase_price_amount,title'])
            ->get();

        $itemCosts = [];
        
        // Group logs by listing first
        $groupedLogs = $fifoLogs->groupBy('listing_id');
        
        $orderItems = collect($order->items);

        foreach ($groupedLogs as $listingId => $logs) {
            $totalLogCost = 0;
            $totalLogQty = 0;
            $batches = [];
            
            foreach ($logs as $log) {
                $batchCost = $log->batch->cost_price ?? $log->listing->purchase_price_amount ?? 0;
                $quantity = abs($log->change_amount);
                $logCost = $quantity * $batchCost;
                
                $totalLogCost += $logCost;
                $totalLogQty += $quantity;
                
                $batches[] = [
                    'quantity' => $quantity,
                    'unit_cost' => $batchCost,
                    'total_chunk_cost' => $logCost
                ];
            }

            // Find matching order item to verify quantity
            $orderItem = $orderItems->firstWhere('id', $listingId);
            
            // Try 'quantity', then 'qty', then default to 1 (same as GeminiService behavior)
            $soldQty = $orderItem['quantity'] ?? $orderItem['qty'] ?? 0;
            if ($orderItem && $soldQty == 0) {
                 $soldQty = 1; 
            }
            
            $finalCost = $totalLogCost;
            $note = null;

            // Anomaly Correction: If logs show more deducted than sold, normalize cost
            if ($soldQty > 0 && $totalLogQty > $soldQty) {
                // Calculate average unit cost from the logs
                $avgUnitCost = $totalLogQty > 0 ? ($totalLogCost / $totalLogQty) : 0;
                // Apply average cost only to the quantity actually sold
                $finalCost = $avgUnitCost * $soldQty;
                
                $note = "Normalized: Inventory logs showed {$totalLogQty} deducted, but only {$soldQty} sold. Using avg unit cost.";
            }

            $itemCosts[$listingId] = [
                'total_cost' => $finalCost,
                'batches' => $batches,
                'note' => $note
            ];
        }
        
        // Pass order data AND normalized costs to Gemini
        $summary = $geminiService->generateOrderSummary($order->toArray(), $itemCosts);

        // Save for future use
        if (!empty($summary) && !str_contains($summary, 'Unable to generate')) {
            $order->ai_summary = $summary;
            $order->save();
        }

        return response()->json([
            'summary' => $summary,
            'status' => 'success',
            'source' => 'api'
        ]);
    }

    /**
     * Get AI-powered inventory health analysis.
     */
    public function inventoryHealth(Request $request, Business $business, \App\Services\GeminiService $geminiService)
    {
        if (!$business->isMember($request->user())) {
            return response()->json(['message' => 'Unauthorized'], 403);
        }

        // 1. Calculate Sales Velocity (Last 30 days)
        $thirtyDaysAgo = now()->subDays(30);
        
        $salesData = DB::table('inventory_logs')
            ->where('business_id', $business->id)
            ->where('reason', 'LIKE', 'sale%')
            ->where('created_at', '>=', $thirtyDaysAgo)
            ->select('listing_id', DB::raw('SUM(ABS(change_amount)) as total_sold'))
            ->groupBy('listing_id')
            ->get()
            ->keyBy('listing_id');

        // 2. Fetch all active listings
        $listings = Listing::where('business_id', $business->id)
            ->whereNull('deleted_at')
            ->get();

        $anomalies = [
            'stockouts' => [],
            'low_stock' => [], // Days remaining < 7
            'dead_stock' => [], // 0 sales and > 0 stock
            'overstock' => [], // > 3 months of supply
            'movers_fast' => [],
            'movers_slow' => [],
        ];

        $totalCapitalTiedUp = 0;
        $totalMonthlyVelocity = 0;
        
        $analysis = $listings->map(function ($listing) use ($salesData, &$anomalies, &$totalCapitalTiedUp, &$totalMonthlyVelocity) {
            $sold = $salesData->get($listing->id)->total_sold ?? 0;
            $velocity = $sold / 30; // units per day
            $totalMonthlyVelocity += $sold;
            
            $stock = $listing->quantity + ($listing->store_quantity ?? 0);
            $cost = $listing->purchase_price_amount ?? 0;
            $totalCapitalTiedUp += ($stock * $cost);

            // Prediction: Days remaining
            $daysRemaining = $velocity > 0 ? floor($stock / $velocity) : 999;
            
            $item = [
                'id' => $listing->id,
                'title' => $listing->title,
                'stock' => $stock,
                'velocity' => round($velocity, 2),
                'days_remaining' => $daysRemaining,
                'capital_tied' => $stock * $cost,
            ];

            if ($stock <= 0) {
                $anomalies['stockouts'][] = $item;
            } elseif ($daysRemaining < 7) {
                $anomalies['low_stock'][] = $item;
            }

            if ($sold == 0 && $stock > 0) {
                $anomalies['dead_stock'][] = $item;
            } elseif ($velocity > 0 && ($stock / ($velocity * 30)) > 3) {
                $anomalies['overstock'][] = $item;
            }

            if ($velocity > 0) {
                if ($velocity > 1) { // 1+ unit per day
                    $anomalies['movers_fast'][] = $item;
                } else {
                    $anomalies['movers_slow'][] = $item;
                }
            }

            return $item;
        });

        // 3. Generate Summary & Health Score
        $outOfStockCount = count($anomalies['stockouts']);
        $deadStockCount = count($anomalies['dead_stock']);
        $overstockCount = count($anomalies['overstock']);
        
        // Simple health score algorithm (Start at 100, deduct for issues)
        $healthScore = 100;
        $healthScore -= ($outOfStockCount * 2);
        $healthScore -= ($deadStockCount * 1.5);
        $healthScore -= ($overstockCount * 1);
        $healthScore = max(0, min(100, $healthScore));

        $summary = [
            'health_score' => $healthScore,
            'total_items' => $listings->count(),
            'total_capital_tied_up' => $totalCapitalTiedUp,
            'total_monthly_velocity' => $totalMonthlyVelocity,
            'currency' => $business->currency ?? 'UGX',
            'anomalies_count' => [
                'stockouts' => count($anomalies['stockouts']),
                'low_stock' => count($anomalies['low_stock']),
                'dead_stock' => count($anomalies['dead_stock']),
                'overstock' => count($anomalies['overstock']),
            ]
        ];

        // 4. Check for existing recent report (Weekly Limit)
        $latestReport = \App\Models\InventoryReport::where('business_id', $business->id)
            ->latest()
            ->first();

        $now = now();
        $nextUpdateDate = $latestReport ? $latestReport->created_at->addDays(7) : $now;
        $canGenerate = !$latestReport || $now->greaterThanOrEqualTo($nextUpdateDate);
        
        // Return cached report if we cannot generate a new one
        if (!$canGenerate) {
            $daysRemaining = $now->diffInDays($nextUpdateDate, false);
            // If less than 24 hours, show hours
            $timeRemainingString = $daysRemaining > 0 
                ? "$daysRemaining days" 
                : ceil($now->diffInHours($nextUpdateDate, false)) . " hours";

            return response()->json([
                'summary' => $latestReport->summary,
                'ai_insights' => $latestReport->ai_insights,
                'anomalies' => $latestReport->anomalies,
                'status' => 'success',
                'source' => 'cache_db',
                'next_update_in' => $timeRemainingString,
                'generated_at' => $latestReport->created_at->toIso8601String(),
            ]);
        }

        // 5. Get AI Strategic Analysis
        $aiInsights = $geminiService->generateInventoryInsights($summary, $anomalies);
        
        // Save the new report
        if (!empty($aiInsights) && !str_contains($aiInsights, 'unavailable')) {
             \App\Models\InventoryReport::create([
                'business_id' => $business->id,
                'summary' => $summary,
                'anomalies' => $anomalies,
                'ai_insights' => $aiInsights,
            ]);
        }



        return response()->json([
            'summary' => $summary,
            'ai_insights' => $aiInsights,
            'anomalies' => $anomalies,
            'status' => 'success',
            'source' => 'live',
             // Next update is 7 days from now
            'next_update_in' => '7 days',
            'generated_at' => now()->toIso8601String(),
        ]);
    }

    /**
     * Fulfill a Layaway order (record payment and potentially complete).
     */
    public function fulfillOrder(Request $request, Business $business, PosOrder $order)
    {
        if (!$business->isMember($request->user())) {
             return response()->json(['message' => 'Unauthorized'], 403);
        }

        if ($order->business_id != $business->id) {
            return response()->json(['message' => 'Order not found for this business'], 404);
        }

        $validator = Validator::make($request->all(), [
            'amount' => 'required|numeric|min:0',
            'method' => 'required|string',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        $amount = (float)$request->amount;

        $totalPaid = $order->payments()->sum('amount');
        $remaining = $order->total_amount - $totalPaid;

        if ($amount > $remaining + 0.5) { // 0.5 buffer for rounding
            return response()->json(['message' => "Payment amount ($amount) exceeds remaining balance ($remaining)"], 422);
        }

        DB::beginTransaction();
        try {
            // Record Payment only if amount > 0
            if ($amount > 0) {
                $order->payments()->create([
                    'payment_method' => $request->method,
                    'amount' => $amount,
                ]);
            }

            // Check if fully paid
            $newTotalPaid = $totalPaid + $amount;
            if ($newTotalPaid >= ($order->total_amount - 1.0)) { // 1.0 buffer for rounding issues
                $order->status = 'completed';
                $order->save();
            }

            DB::commit();

            return response()->json([
                'status' => 'success',
                'order_status' => $order->status,
                'message' => 'Payment recorded successfully' . ($order->status == 'completed' ? '. Order Completed.' : '.')
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json(['message' => 'Failed to record payment', 'error' => $e->getMessage()], 500);
        }
    }
}
