<?php

namespace App\Http\Controllers\Api\V1;

use App\Http\Controllers\Controller;
use App\Http\Resources\ApiResponse;
use App\Models\Business;
use App\Models\Listing;
use App\Services\ListingViewTrackerService;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Gate;

class ListingViewsController extends Controller
{
    protected ListingViewTrackerService $viewTracker;

    public function __construct(ListingViewTrackerService $viewTracker)
    {
        $this->viewTracker = $viewTracker;
    }

    /**
     * Get view statistics for a listing
     * Owner/manager can see detailed stats
     */
    public function stats(Business $business, Listing $listing): JsonResponse
    {
        Gate::authorize('manageListings', $business);

        if ($listing->business_id !== $business->id) {
            return ApiResponse::error('Listing does not belong to this business', 403);
        }

        $stats = $this->viewTracker->getViewStats($listing);

        return ApiResponse::success($stats);
    }

    /**
     * Get list of viewers for a listing
     * Owner/manager can see who viewed their listing
     */
    public function viewers(Business $business, Listing $listing): JsonResponse
    {
        Gate::authorize('manageListings', $business);

        if ($listing->business_id !== $business->id) {
            return ApiResponse::error('Listing does not belong to this business', 403);
        }

        $viewers = $this->viewTracker->getViewers($listing);

        return ApiResponse::success([
            'viewers' => $viewers,
            'total_count' => count($viewers),
        ]);
    }
}
