<?php

namespace App\Http\Controllers\Api\V1;

use App\Http\Controllers\Controller;
use App\Http\Resources\ApiResponse;
use App\Models\Listing;
use App\Services\PremiumService;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class ListingPromotionController extends Controller
{
    protected PremiumService $premiumService;

    public function __construct(PremiumService $premiumService)
    {
        $this->premiumService = $premiumService;
    }

    public function featureListing(Request $request, int $listingId): JsonResponse
    {
        $request->validate([
            'payment_method' => 'nullable|string',
            'transaction_id' => 'nullable|string',
            'amount' => 'nullable|numeric|min:0',
            'duration_days' => 'nullable|integer|min:1|max:365',
        ]);

        $userId = auth('api')->id();
        $listing = Listing::where('id', $listingId)
            ->where('business_id', function ($query) use ($userId) {
                $query->select('id')
                    ->from('businesses')
                    ->whereRaw('businesses.id = listings.business_id')
                    ->where(function ($q) use ($userId) {
                        $q->where('owner_id', $userId)
                          ->orWhereExists(function ($subQ) use ($userId) {
                              $subQ->select('id')
                                  ->from('business_members')
                                  ->whereRaw('business_members.business_id = businesses.id')
                                  ->where('user_id', $userId);
                          });
                    });
            })
            ->first();

        if (!$listing) {
            return ApiResponse::error('Listing not found or you do not have permission', 404);
        }

        $paymentData = null;
        if ($request->filled('payment_method')) {
            $paymentData = [
                'amount' => $request->input('amount'),
                'payment_method' => $request->input('payment_method'),
                'transaction_id' => $request->input('transaction_id'),
                'duration_days' => $request->input('duration_days', 30),
            ];
        }

        $result = $this->premiumService->featureListing($listingId, $userId, $paymentData);

        if (!$result['success']) {
            return ApiResponse::error($result['message'], 400);
        }

        return ApiResponse::success($result);
    }

    public function bumpListing(Request $request, int $listingId): JsonResponse
    {
        $request->validate([
            'payment_method' => 'nullable|string',
            'transaction_id' => 'nullable|string',
            'amount' => 'nullable|numeric|min:0',
        ]);

        $userId = auth('api')->id();
        $listing = Listing::where('id', $listingId)
            ->where('business_id', function ($query) use ($userId) {
                $query->select('id')
                    ->from('businesses')
                    ->whereRaw('businesses.id = listings.business_id')
                    ->where(function ($q) use ($userId) {
                        $q->where('owner_id', $userId)
                          ->orWhereExists(function ($subQ) use ($userId) {
                              $subQ->select('id')
                                  ->from('business_members')
                                  ->whereRaw('business_members.business_id = businesses.id')
                                  ->where('user_id', $userId);
                          });
                    });
            })
            ->first();

        if (!$listing) {
            return ApiResponse::error('Listing not found or you do not have permission', 404);
        }

        $paymentData = null;
        if ($request->filled('payment_method')) {
            $paymentData = [
                'amount' => $request->input('amount'),
                'payment_method' => $request->input('payment_method'),
                'transaction_id' => $request->input('transaction_id'),
            ];
        }

        $result = $this->premiumService->bumpListing($listingId, $userId, $paymentData);

        if (!$result['success']) {
            return ApiResponse::error($result['message'], 400);
        }

        return ApiResponse::success($result);
    }

    public function myPromotions(Request $request): JsonResponse
    {
        $userId = auth('api')->id();
        $type = $request->query('type');

        $promotions = $this->premiumService->getUserPromotions($userId, $type);

        return ApiResponse::success([
            'promotions' => $promotions,
            'total' => $promotions->count(),
        ]);
    }

    public function checkEligibility(int $listingId): JsonResponse
    {
        $userId = auth('api')->id();
        $subscription = $this->premiumService->getActiveSubscription($userId);

        $canFeature = false;
        $canBump = false;
        $method = 'payment_required';

        if ($subscription) {
            $canFeature = $subscription->canFeatureListing();
            $canBump = $subscription->canBumpListing();
            $method = 'subscription';
        }

        return ApiResponse::success([
            'can_feature' => $canFeature,
            'can_bump' => $canBump,
            'method' => $method,
            'subscription' => $subscription,
            'feature_remaining' => $subscription ? ($subscription->package->max_featured_listings - $subscription->featured_listings_used) : 0,
            'bumps_remaining' => $subscription ? ($subscription->package->max_bumps_per_month - $subscription->bumps_used_this_month) : 0,
        ]);
    }
}
