<?php

namespace App\Http\Controllers\Api\V1;

use App\Http\Controllers\Controller;
use App\Http\Resources\ApiResponse;
use App\Models\Business;
use App\Models\Listing;
use App\Services\ListingAnalyticsService;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Gate;

class ListingAnalyticsController extends Controller
{
    protected ListingAnalyticsService $analytics;

    public function __construct(ListingAnalyticsService $analytics)
    {
        $this->analytics = $analytics;
    }

    /**
     * Track impression (listing shown in feed)
     * Called from frontend when listing appears in viewport
     */
    public function trackImpression(Request $request): JsonResponse
    {
        $request->validate([
            'listing_id' => 'required|integer|exists:listings,id',
            'source' => 'nullable|string|in:feed,search,category,related,business_profile',
            'context' => 'nullable|array',
        ]);

        $tracked = $this->analytics->trackImpression(
            $request->input('listing_id'),
            $request->input('source'),
            $request->input('context'),
            $request
        );

        return ApiResponse::success(['tracked' => $tracked]);
    }

    /**
     * Track click (user clicked listing from feed)
     * Called from frontend when user clicks listing card
     */
    public function trackClick(Request $request): JsonResponse
    {
        $request->validate([
            'listing_id' => 'required|integer|exists:listings,id',
            'source' => 'nullable|string|in:feed,search,category,related',
            'context' => 'nullable|array',
        ]);

        $tracked = $this->analytics->trackClick(
            $request->input('listing_id'),
            $request->input('source'),
            $request->input('context'),
            $request
        );

        return ApiResponse::success(['tracked' => $tracked]);
    }

    /**
     * Track interaction (phone reveal, WhatsApp, share, etc.)
     */
    public function trackInteraction(Request $request): JsonResponse
    {
        $request->validate([
            'listing_id' => 'required|integer|exists:listings,id',
            'interaction_type' => 'required|string|in:phone_reveal,whatsapp_click,email_click,share,favorite,report',
            'metadata' => 'nullable|array',
        ]);

        $tracked = $this->analytics->trackInteraction(
            $request->input('listing_id'),
            $request->input('interaction_type'),
            $request->input('metadata'),
            $request
        );

        return ApiResponse::success(['tracked' => $tracked]);
    }

    /**
     * Get comprehensive analytics for a listing (owner/manager only)
     */
    public function getListingAnalytics(Business $business, Listing $listing, Request $request): JsonResponse
    {
        Gate::authorize('manageListings', $business);

        if ($listing->business_id != $business->id) {
            return ApiResponse::error('Listing does not belong to this business', 403);
        }

        $period = $request->query('period', '30days');
        $analytics = $this->analytics->getAnalytics($listing, $period);

        return ApiResponse::success($analytics);
    }

    /**
     * Get platform-wide analytics (admin only or public summary for advertisers)
     */
    public function getPlatformAnalytics(Request $request): JsonResponse
    {
        // For now, make this public to attract advertisers
        // In production, you might want to restrict detailed data to admins
        
        $category = $request->query('category');
        $period = $request->query('period', '30days');

        $analytics = $this->analytics->getPlatformAnalytics($category, $period);

        return ApiResponse::success($analytics);
    }
}
