<?php

namespace App\Http\Controllers\Api\V1;

use App\Http\Controllers\Controller;
use App\Models\Business;
use App\Models\Listing;
use App\Services\InventoryService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;

class InventoryController extends Controller
{
    protected $inventoryService;

    public function __construct(InventoryService $inventoryService)
    {
        $this->inventoryService = $inventoryService;
    }

    /**
     * Add stock to Store or Shop.
     */
    public function addStock(Request $request, Business $business, Listing $listing)
    {
        // 1. Authorize (Listing must belong to business, user must be member)
        if ((int) $listing->business_id !== (int) $business->id) {
            return response()->json(['message' => 'Listing does not belong to this business'], 403);
        }
        if (!$business->isMember($request->user())) {
            return response()->json(['message' => 'Unauthorized'], 403);
        }

        // 2. Validate
        $validator = Validator::make($request->all(), [
            'quantity' => 'required|integer|min:1',
            'cost_price' => 'required|numeric|min:0',
            'supplier' => 'nullable|string|max:255',
            'reference_note' => 'nullable|string|max:255',
            'location' => 'required|in:store,shop',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        // 3. Process
        try {
            $batch = $this->inventoryService->addStock(
                $listing,
                $request->quantity,
                $request->cost_price,
                $request->supplier,
                $request->reference_note,
                $request->user()->id,
                $request->location
            );

            return response()->json([
                'message' => 'Stock added successfully',
                'batch' => $batch,
                'listing' => $this->transformListingForPos($listing->refresh()),
            ]);
        } catch (\Exception $e) {
            return response()->json(['message' => 'Failed to add stock: ' . $e->getMessage()], 500);
        }
    }

    /**
     * Transfer stock from Store to Shop.
     */
    public function transferStock(Request $request, Business $business, Listing $listing)
    {
        if ((int) $listing->business_id !== (int) $business->id) {
            return response()->json(['message' => 'Listing does not belong to this business'], 403);
        }
        if (!$business->isMember($request->user())) {
            return response()->json(['message' => 'Unauthorized'], 403);
        }

        $validator = Validator::make($request->all(), [
            'quantity' => 'required|integer|min:1',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        try {
            $this->inventoryService->moveStockToShop(
                $listing,
                $request->quantity,
                $request->user()->id
            );

            return response()->json([
                'message' => 'Stock transferred successfully',
                'listing' => $this->transformListingForPos($listing->refresh()),
            ]);
        } catch (\Exception $e) {
            return response()->json(['message' => $e->getMessage()], 400); // 400 for logic errors (e.g., insufficient stock)
        }
    }

    /**
     * Remove stock (manual adjustment).
     */
    public function removeStock(Request $request, Business $business, Listing $listing)
    {
        if ((int) $listing->business_id !== (int) $business->id) {
            return response()->json(['message' => 'Listing does not belong to this business'], 403);
        }
        if (!$business->isMember($request->user())) {
            return response()->json(['message' => 'Unauthorized'], 403);
        }

        $validator = Validator::make($request->all(), [
            'quantity' => 'required|integer|min:1',
            'reason' => 'required|string|max:255',
            'location' => 'required|in:store,shop',
            'reference_note' => 'nullable|string|max:255',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        try {
            $this->inventoryService->removeStock(
                $listing,
                $request->quantity,
                $request->location,
                $request->reason,
                $request->reference_note,
                $request->user()->id
            );

            return response()->json([
                'message' => 'Stock removed successfully',
                'listing' => $this->transformListingForPos($listing->refresh()),
            ]);
        } catch (\Exception $e) {
            return response()->json(['message' => $e->getMessage()], 400);
        }
    }

    /**
     * Get inventory history for a specific listing.
     */
    public function getLogs(Business $business, Listing $listing)
    {
        if ((int) $listing->business_id !== (int) $business->id) {
            return response()->json(['message' => 'Listing does not belong to this business'], 403);
        }

        $logs = $listing->inventoryLogs()
            ->with('user:id,full_name')
            ->orderBy('created_at', 'desc')
            ->paginate(100);

        return response()->json($logs);
    }

    /**
     * Transform listing to POS format.
     */
    private function transformListingForPos(Listing $listing)
    {
        return [
            'id' => $listing->id,
            'title' => $listing->title,
            'price' => $listing->price_amount,
            'stock' => $listing->quantity,
            'store_stock' => $listing->store_quantity ?? 0,
            'sku' => $listing->sku,
            'barcode' => $listing->barcode,
            'is_deleted' => !is_null($listing->deleted_at),
            'quick_access' => (bool)$listing->pos_quick_access,
            'updated_at' => $listing->updated_at?->timestamp ?? now()->timestamp,
            'category' => $listing->category,
            'thumbnail_url' => $listing->thumbnail_url,
            'cost_price' => $listing->purchase_price_amount,
        ];
    }
}
