<?php

namespace App\Http\Controllers\Api\V1;

use App\Http\Controllers\Controller;
use App\Http\Resources\ApiResponse;
use App\Http\Resources\BusinessResource;
use App\Http\Resources\ListingResource;
use App\Models\Business;
use App\Models\Listing;
use App\Models\UserFavoriteBusiness;
use App\Models\UserFavoriteListing;
use App\Models\UserFavoriteSearch;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class FavoritesController extends Controller
{
    public function addBusinessToFavorites(Request $request): JsonResponse
    {
        $request->validate([
            'business_id' => 'required|integer|exists:businesses,id',
        ]);

        $userId = auth('api')->id();
        $businessId = $request->input('business_id');

        $favorite = UserFavoriteBusiness::firstOrCreate([
            'user_id' => $userId,
            'business_id' => $businessId,
        ]);

        return ApiResponse::success([
            'message' => $favorite->wasRecentlyCreated ? 'Business added to favorites' : 'Business already in favorites',
            'favorite' => $favorite,
        ]);
    }

    public function removeBusinessFromFavorites(int $businessId): JsonResponse
    {
        $userId = auth('api')->id();

        $deleted = UserFavoriteBusiness::where('user_id', $userId)
            ->where('business_id', $businessId)
            ->delete();

        if (!$deleted) {
            return ApiResponse::error('Business not found in favorites', 404);
        }

        return ApiResponse::success([
            'message' => 'Business removed from favorites',
        ]);
    }

    public function getFavoriteBusinesses(): JsonResponse
    {
        $userId = auth('api')->id();

        $favorites = UserFavoriteBusiness::where('user_id', $userId)
            ->with('business')
            ->latest()
            ->get();

        $businesses = $favorites->map(function ($favorite) {
            return new BusinessResource($favorite->business);
        });

        return ApiResponse::success([
            'favorites' => $businesses,
            'total' => $favorites->count(),
        ]);
    }

    public function addListingToFavorites(Request $request): JsonResponse
    {
        $request->validate([
            'listing_id' => 'required|integer|exists:listings,id',
        ]);

        $userId = auth('api')->id();
        $listingId = $request->input('listing_id');

        $favorite = UserFavoriteListing::firstOrCreate([
            'user_id' => $userId,
            'listing_id' => $listingId,
        ]);

        return ApiResponse::success([
            'message' => $favorite->wasRecentlyCreated ? 'Listing added to favorites' : 'Listing already in favorites',
            'favorite' => $favorite,
        ]);
    }

    public function removeListingFromFavorites(int $listingId): JsonResponse
    {
        $userId = auth('api')->id();

        $deleted = UserFavoriteListing::where('user_id', $userId)
            ->where('listing_id', $listingId)
            ->delete();

        if (!$deleted) {
            return ApiResponse::error('Listing not found in favorites', 404);
        }

        return ApiResponse::success([
            'message' => 'Listing removed from favorites',
        ]);
    }

    public function getFavoriteListings(): JsonResponse
    {
        $userId = auth('api')->id();

        $favorites = UserFavoriteListing::where('user_id', $userId)
            ->with(['listing.business', 'listing.media'])
            ->latest()
            ->get();

        $listings = $favorites->map(function ($favorite) {
            return new ListingResource($favorite->listing);
        });

        return ApiResponse::success([
            'favorites' => $listings,
            'total' => $favorites->count(),
        ]);
    }

    public function saveFavoriteSearch(Request $request): JsonResponse
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'search_type' => 'sometimes|string|in:text,image,category,location',
            'query' => 'nullable|string',
            'filters' => 'nullable|array',
        ]);

        $userId = auth('api')->id();

        $favoriteSearch = UserFavoriteSearch::create([
            'user_id' => $userId,
            'name' => $request->input('name'),
            'search_type' => $request->input('search_type', 'text'),
            'query' => $request->input('query'),
            'filters' => $request->input('filters'),
        ]);

        return ApiResponse::success([
            'message' => 'Search saved to favorites',
            'favorite_search' => $favoriteSearch,
        ], 201);
    }

    public function updateFavoriteSearch(Request $request, int $id): JsonResponse
    {
        $request->validate([
            'name' => 'sometimes|string|max:255',
            'search_type' => 'sometimes|string|in:text,image,category,location',
            'query' => 'nullable|string',
            'filters' => 'nullable|array',
        ]);

        $userId = auth('api')->id();

        $favoriteSearch = UserFavoriteSearch::where('user_id', $userId)
            ->where('id', $id)
            ->first();

        if (!$favoriteSearch) {
            return ApiResponse::error('Favorite search not found', 404);
        }

        $favoriteSearch->update($request->only(['name', 'search_type', 'query', 'filters']));

        return ApiResponse::success([
            'message' => 'Favorite search updated',
            'favorite_search' => $favoriteSearch,
        ]);
    }

    public function deleteFavoriteSearch(int $id): JsonResponse
    {
        $userId = auth('api')->id();

        $deleted = UserFavoriteSearch::where('user_id', $userId)
            ->where('id', $id)
            ->delete();

        if (!$deleted) {
            return ApiResponse::error('Favorite search not found', 404);
        }

        return ApiResponse::success([
            'message' => 'Favorite search deleted',
        ]);
    }

    public function getFavoriteSearches(): JsonResponse
    {
        $userId = auth('api')->id();

        $favoriteSearches = UserFavoriteSearch::where('user_id', $userId)
            ->latest()
            ->get();

        return ApiResponse::success([
            'favorite_searches' => $favoriteSearches,
            'total' => $favoriteSearches->count(),
        ]);
    }

    public function checkIfFavorited(Request $request): JsonResponse
    {
        $request->validate([
            'type' => 'required|string|in:business,listing',
            'id' => 'required|integer',
        ]);

        $userId = auth('api')->id();
        $type = $request->input('type');
        $id = $request->input('id');

        $isFavorited = false;

        if ($type === 'business') {
            $isFavorited = UserFavoriteBusiness::where('user_id', $userId)
                ->where('business_id', $id)
                ->exists();
        } elseif ($type === 'listing') {
            $isFavorited = UserFavoriteListing::where('user_id', $userId)
                ->where('listing_id', $id)
                ->exists();
        }

        return ApiResponse::success([
            'is_favorited' => $isFavorited,
        ]);
    }
}
