<?php

namespace App\Http\Controllers\Api\V1;

use App\Http\Controllers\Controller;
use App\Http\Resources\ApiResponse;
use App\Http\Resources\ConversationResource;
use App\Http\Resources\MessageResource;
use App\Models\Business;
use App\Models\Conversation;
use App\Models\Listing;
use App\Models\Message;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class ChatController extends Controller
{
    /**
     * Get the user's inbox (Buying or Selling).
     */
    public function index(Request $request): JsonResponse
    {
        $user = auth('api')->user();
        $type = $request->query('type', 'buying'); // 'buying' or 'selling'

        $query = Conversation::query()
            ->with(['listing.media', 'listing.business', 'buyer', 'business', 'latestMessage'])
            ->orderByDesc('last_message_at');

        if ($type === 'selling') {
            // Get conversations for businesses where the user is a member/owner
            $businessIds = Business::where('owner_user_id', $user->id)
                ->orWhereHas('memberships', function ($q) use ($user) {
                    $q->where('user_id', $user->id);
                })
                ->pluck('id');

            $query->whereIn('business_id', $businessIds);
        } else {
            // Default: Buying conversations
            $query->where('buyer_id', $user->id);
        }

        $conversations = $query->paginate(20);

        return ApiResponse::success([
            'conversations' => ConversationResource::collection($conversations->items()),
            'pagination' => [
                'current_page' => $conversations->currentPage(),
                'per_page' => $conversations->perPage(),
                'total' => $conversations->total(),
                'last_page' => $conversations->lastPage(),
            ],
        ]);
    }

    /**
     * Get messages for a specific conversation.
     */
    public function show(Conversation $conversation): JsonResponse
    {
        $user = auth('api')->user();

        // Security: Check if user is the buyer or a member of the business
        if (!$this->canAccessConversation($user, $conversation)) {
            return ApiResponse::forbidden('You do not have access to this conversation.');
        }

        $messages = $conversation->messages()
            ->with('sender')
            ->orderByDesc('created_at')
            ->paginate(50);

        // Mark messages as read
        $conversation->messages()
            ->where('sender_id', '!=', $user->id)
            ->whereNull('read_at')
            ->update(['read_at' => now()]);

        return ApiResponse::success([
            'messages' => MessageResource::collection($messages->items()),
            'pagination' => [
                'current_page' => $messages->currentPage(),
                'per_page' => $messages->perPage(),
                'total' => $messages->total(),
                'last_page' => $messages->lastPage(),
            ],
        ]);
    }

    /**
     * Send a message.
     */
    public function store(Request $request): JsonResponse
    {
        $user = auth('api')->user();

        // Debug log to see what's coming in
        \Log::info('Chat store request:', [
            'listing_id' => $request->listing_id,
            'conversation_id' => $request->conversation_id,
            'message_len' => strlen($request->message),
            'has_attachment' => $request->hasFile('attachment'),
            'type' => $request->type
        ]);

        $validator = Validator::make($request->all(), [
            'listing_id' => 'nullable|exists:listings,id',
            'conversation_id' => 'nullable|exists:conversations,id',
            'message' => 'nullable|string|max:2000',
            'attachment' => 'nullable|image|max:10240', // Increase to 10MB just in case
            'type' => 'nullable|string|in:text,image,location',
        ]);

        if ($validator->fails()) {
            return ApiResponse::validationError($validator->errors()->toArray());
        }

        $listingId = $request->listing_id;
        $conversationId = $request->conversation_id;

        // Ensure at least one is provided
        if (!$listingId && !$conversationId) {
            return ApiResponse::validationError(['listing_id' => ['Either listing_id or conversation_id is required.']]);
        }

        if ($conversationId) {
            $conversation = Conversation::findOrFail($conversationId);
        } else {
            // Find or create conversation for this listing and buyer
            $listing = Listing::findOrFail($listingId);
            $conversation = Conversation::firstOrCreate([
                'listing_id' => $listing->id,
                'buyer_id' => $user->id,
                'business_id' => $listing->business_id,
            ]);
        }

        // Security check
        if (!$this->canAccessConversation($user, $conversation)) {
            return ApiResponse::forbidden('You do not have access to this conversation.');
        }

        $attachmentPath = null;
        $type = $request->type ?? 'text';

        if ($request->hasFile('attachment')) {
            $file = $request->file('attachment');
            $filename = time() . '_' . uniqid() . '.' . $file->getClientOriginalExtension();
            $attachmentPath = $file->storeAs('chat/attachments', $filename, 'public');
            $type = 'image';
        }

        // Ensure we have either message or attachment
        if (!$attachmentPath && empty($request->message)) {
            return ApiResponse::validationError(['message' => ['Message or attachment is required.']]);
        }

        // Create message
        $message = Message::create([
            'conversation_id' => $conversation->id,
            'sender_id' => $user->id,
            'message' => $request->message ?? '',
            'attachment' => $attachmentPath,
            'type' => $type,
        ]);

        // Update conversation last activity
        $conversation->update(['last_message_at' => now()]);

        // Trigger Push & Email Notification
        try {
            $fcmService = resolve(\App\Services\FcmService::class);
            $senderName = $user->full_name ?? $user->username ?? 'Someone';
            $title = "You have a new message from {$senderName}";
            $body = ($type === 'image') ? 'Sent an image' : ($message->message ?? 'New message');
            
            $notificationData = [
                'type' => 'chat',
                'conversation_id' => (string) $conversation->id,
                'sender_id' => (string) $user->id,
                'listing_id' => (string) $conversation->listing_id,
                'sender_name' => (string) ($user->full_name ?? $user->username),
            ];

            if ($user->id == $conversation->buyer_id) {
                // Buyer sent message to business - Notify business team
                
                // 1. Push
                $fcmService->sendToBusiness($conversation->business_id, $title, $body, $notificationData);

                // 2. Email (Get all recipients like FcmService does)
                $business = \App\Models\Business::with(['memberships.user', 'owner'])->find($conversation->business_id);
                if ($business) {
                    $recipients = collect();
                    if ($business->owner) $recipients->push($business->owner);
                    foreach ($business->memberships as $membership) {
                        if ($membership->user) $recipients->push($membership->user);
                    }
                    
                    // Filter duplicates
                    $uniqueRecipients = $recipients->unique('id');

                    foreach ($uniqueRecipients as $recipient) {
                        if ($recipient->id != $user->id) { 
                            // Email
                            if ($recipient->email) {
                                \Mail::to($recipient->email)->queue(
                                    new \App\Mail\NewMessageNotification(
                                        $message, 
                                        $user, 
                                        $recipient->full_name ?? $recipient->username ?? 'Business Manager'
                                    )
                                );
                            }

                            // 3. Database Notification (In-App)
                            $recipient->notify(new \App\Notifications\GeneralNotification($title, $body, $notificationData));
                        }
                    }
                }

            } else {
                // Seller/Member sent message to buyer - Notify buyer
                
                // 1. Push
                $fcmService->sendToUser($conversation->buyer_id, $title, $body, $notificationData);

                // 2. Email & 3. Database
                $buyer = \App\Models\User::find($conversation->buyer_id);
                if ($buyer) {
                    if ($buyer->email) {
                        \Mail::to($buyer->email)->queue(
                            new \App\Mail\NewMessageNotification(
                                $message,
                                $user,
                                $buyer->full_name ?? $buyer->username ?? 'User'
                            )
                        );
                    }

                    // Database Notification
                    $buyer->notify(new \App\Notifications\GeneralNotification($title, $body, $notificationData));
                }
            }
        } catch (\Exception $e) {
            \Log::error('Failed to send chat notifications: ' . $e->getMessage());
        }

        return ApiResponse::created(new MessageResource($message->load('sender')), 'Message sent.');
    }

    /**
     * Mark a conversation as read.
     */
    public function markRead(Conversation $conversation): JsonResponse
    {
        $user = auth('api')->user();

        if (!$this->canAccessConversation($user, $conversation)) {
            return ApiResponse::forbidden();
        }

        $conversation->messages()
            ->where('sender_id', '!=', $user->id)
            ->whereNull('read_at')
            ->update(['read_at' => now()]);

        return ApiResponse::success(null, 'Conversation marked as read.');
    }

    /**
     * Helper to verify access.
     */
    private function canAccessConversation($user, Conversation $conversation): bool
    {
        // Is buyer?
        if ($conversation->buyer_id == $user->id) {
            return true;
        }

        // Is business member/owner?
        $isMember = Business::where('id', $conversation->business_id)
            ->where(function ($q) use ($user) {
                $q->where('owner_user_id', $user->id)
                    ->orWhereHas('memberships', function ($mq) use ($user) {
                        $mq->where('user_id', $user->id);
                    });
            })
            ->exists();

        return $isMember;
    }
}
