<?php

namespace App\Http\Controllers\Api\V1;

use App\Http\Controllers\Controller;
use App\Http\Requests\Listing\CreateListingRequest;
use App\Http\Requests\Listing\UpdateListingRequest;
use App\Http\Requests\Listing\UploadListingMediaRequest;
use App\Http\Resources\ApiResponse;
use App\Http\Resources\ListingResource;
use App\Jobs\AutoReviewListing;
use App\Models\Business;
use App\Models\Listing;
use App\Models\ListingMedia;
use App\Models\UserSubscription;
use App\Services\ImageProcessingService;
use App\Services\PremiumService;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class BusinessListingController extends Controller
{
    protected $premiumService;

    public function __construct(PremiumService $premiumService)
    {
        $this->premiumService = $premiumService;
    }

    public function index(Request $request, Business $business): JsonResponse
    {
        Gate::authorize('manageListings', $business);

        $perPage = (int) $request->query('per_page', 20);
        $perPage = max(1, min(100, $perPage));

        $query = Listing::query()->where('business_id', $business->id);

        if ($request->filled('status')) {
            $query->where('status', $request->query('status'));
        }

        if ($request->filled('q')) {
            $q = $request->query('q');
            $query->where(function ($sub) use ($q) {
                $sub->where('title', 'like', "%{$q}%")
                    ->orWhere('description', 'like', "%{$q}%");
            });
        }

        $listings = $query
            ->with(['media'])
            ->orderByDesc('created_at')
            ->paginate($perPage);

        return ApiResponse::success([
            'listings' => ListingResource::collection($listings->items()),
            'pagination' => [
                'current_page' => $listings->currentPage(),
                'per_page' => $listings->perPage(),
                'total' => $listings->total(),
                'last_page' => $listings->lastPage(),
            ],
        ]);
    }

    public function store(CreateListingRequest $request, Business $business): JsonResponse
    {
        Gate::authorize('manageListings', $business);

        $user = auth('api')->user();
        $data = $request->validated();

        if (array_key_exists('tags', $data)) {
            $data['tags'] = $this->normalizeTags($data['tags']);
        }

        // Check subscription eligibility
        $eligibility = $this->premiumService->canBusinessCreateListing($business->id);
        if (!$eligibility['allowed']) {
            return ApiResponse::error($eligibility['reason'], 403);
        }

        // Always set new listings to REVIEW status
        $status = Listing::STATUS_REVIEW;

        $listing = Listing::create([
            'business_id' => $business->id,
            'created_by_user_id' => $user?->id,
            'category' => $data['category'] ?? null,
            'sub_category' => $data['sub_category'] ?? null,
            'sub_sub_category' => $data['sub_sub_category'] ?? null,
            'title' => $data['title'],
            'description' => $data['description'],
            'price_amount' => $data['price_amount'] ?? null,
            'purchase_price_amount' => $data['purchase_price_amount'] ?? null,
            'currency' => $data['currency'] ?? 'UGX',
            'status' => $status,
            'quantity' => 0,
            'sku' => $data['sku'] ?? null,
            'barcode' => $data['barcode'] ?? null,
            'allow_pos' => (bool) ($data['allow_pos'] ?? false),
            'location_country_code' => $data['location_country_code'] ?? null,
            'location_city' => $data['location_city'] ?? null,
            'location_district' => $data['location_district'] ?? null,
            'latitude' => $data['latitude'] ?? null,
            'longitude' => $data['longitude'] ?? null,
            'attributes' => $data['attributes'] ?? null,
            'custom_fields' => $data['custom_fields'] ?? null,
            'tags' => $this->normalizeTags($data['tags'] ?? null),
            'meta_title' => $data['meta_title'] ?? null,
            'meta_description' => $data['meta_description'] ?? null,
            'meta_keywords' => $data['meta_keywords'] ?? null,
            'canonical_url' => $data['canonical_url'] ?? null,
            'expires_at' => $data['expires_at'] ?? null,
            'published_at' => in_array($status, [Listing::STATUS_PUBLISHED, Listing::STATUS_REVIEW], true) ? now() : null,
            'bumped_at' => in_array($status, [Listing::STATUS_PUBLISHED, Listing::STATUS_REVIEW], true) ? now() : null,
        ]);

        // Handle pre-uploaded media via media_ids
        if ($request->filled('media_ids')) {
            $mediaIds = (array) $request->input('media_ids');
            $primaryIndex = $request->input('primary_media_index');

            $preUploadedMedia = ListingMedia::query()
                ->whereIn('id', $mediaIds)
                ->whereNull('listing_id')
                ->where('uploaded_by_user_id', $user?->id)
                ->get();

            foreach ($preUploadedMedia as $i => $media) {
                $media->update([
                    'listing_id' => $listing->id,
                    'sort_order' => $i,
                    'is_primary' => $primaryIndex !== null ? ((int) $primaryIndex === (int) $i) : ($i === 0),
                ]);
            }
        }
        // Fallback: Handle direct file uploads (backward compatibility)
        elseif ($request->hasFile('media')) {
            $files = (array) $request->file('media');
            $primaryIndex = $request->input('primary_media_index');
            $imageService = new ImageProcessingService();

            foreach ($files as $i => $file) {
                if (!$file) {
                    continue;
                }

                $mime = (string) $file->getClientMimeType();
                $type = str_starts_with($mime, 'video/') ? ListingMedia::TYPE_VIDEO : ListingMedia::TYPE_IMAGE;
                $contentHash = hash_file('sha256', $file->getRealPath());
                $path = $file->store('listing-media', 'public');

                $watermarkedPath = null;
                $tags = [];
                
                // Process images (compress + watermark + tag detection)
                if ($type === ListingMedia::TYPE_IMAGE && str_starts_with($mime, 'image/')) {
                    $processed = $imageService->processImage($path, $business->name);
                    $path = $processed['compressed_path'];
                    $watermarkedPath = $processed['watermarked_path'];
                    $tags = $processed['tags'] ?? [];
                }

                ListingMedia::create([
                    'listing_id' => $listing->id,
                    'uploaded_by_user_id' => $user?->id,
                    'path' => $path,
                    'watermarked_path' => $watermarkedPath,
                    'tags' => $tags,
                    'mime_type' => $mime,
                    'size_bytes' => $file->getSize(),
                    'content_hash' => $contentHash,
                    'type' => $type,
                    'sort_order' => $i,
                    'is_primary' => $primaryIndex !== null ? ((int) $primaryIndex === (int) $i) : ($i === 0),
                ]);
            }
        }

        $listing = Listing::query()
            ->whereKey($listing->getKey())
            ->with(['business', 'media'])
            ->firstOrFail();

        if ($listing->status === Listing::STATUS_REVIEW) {
            AutoReviewListing::dispatch($listing->id)->delay(now()->addMinutes(rand(1, 2)));
        }

        // Increment listing usage if it's within a subscription
        $this->premiumService->incrementListingUsage($business->id, $listing->id);

        return ApiResponse::created(new ListingResource($listing), 'Listing created successfully.');
    }

    public function show(Business $business, Listing $listing): JsonResponse
    {
        Gate::authorize('manageListings', $business);

        if ((int) $listing->business_id !== (int) $business->id) {
            return ApiResponse::notFound('Listing not found for this business.');
        }

        $listing = Listing::query()
            ->whereKey($listing->getKey())
            ->with(['business', 'media'])
            ->firstOrFail();

        return ApiResponse::success(new ListingResource($listing));
    }

    public function update(UpdateListingRequest $request, Business $business, Listing $listing): JsonResponse
    {
        Gate::authorize('manageListings', $business);

        if ((int) $listing->business_id !== (int) $business->id) {
            return ApiResponse::notFound('Listing not found for this business.');
        }

        $data = $request->validated();

        if (array_key_exists('tags', $data)) {
            $data['tags'] = $this->normalizeTags($data['tags']);
        }

        // Fields that trigger a re-review if changed
        $reviewRequiredFields = [
            'title', 'description', 'category', 'sub_category', 'sub_sub_category',
            'price_amount', 'currency', 'media_ids', 'custom_fields', 'attributes', 
            'location_city', 'location_district', 'location_country_code', 'tags'
        ];

        $needsReview = false;
        foreach ($reviewRequiredFields as $field) {
            if (array_key_exists($field, $data)) {
                // media_ids is a special field used for relationship updates, not an attribute
                if ($field === 'media_ids') {
                    $needsReview = true; // Assume media changes always require review for now
                    break;
                }

                $currentValue = $listing->$field;
                $newValue = $data[$field];
                
                if (is_array($currentValue) || is_array($newValue)) {
                    if (json_encode($currentValue) !== json_encode($newValue)) {
                        $needsReview = true;
                        break;
                    }
                } elseif ((string)$currentValue !== (string)$newValue) {
                    $needsReview = true;
                    break;
                }
            }
        }

        // If updating a published or rejected listing, move it back to review
        // unless expiring it or only making minor updates.
        if (isset($data['status']) && $data['status'] === Listing::STATUS_EXPIRED) {
            // Allow expiring
            $data['expires_at'] = now();
        } elseif ($needsReview) {
            // Force review on significant update
            $data['status'] = Listing::STATUS_REVIEW;
        } elseif (isset($data['status']) && $data['status'] === Listing::STATUS_REVIEW) {
            // If they explicitly sent 'review' but no significant changes were detected:
            // 1. If it's already published/rejected, ignore it (prevent accidental re-review)
            // 2. If it's draft, allow it (moving from draft to review)
            if (in_array($listing->status, [Listing::STATUS_PUBLISHED, Listing::STATUS_REJECTED, Listing::STATUS_SOLD, Listing::STATUS_EXPIRED])) {
                unset($data['status']);
            }
        }

        if (array_key_exists('status', $data) && $data['status'] === Listing::STATUS_PUBLISHED) {
             // Block explicit publish from API, must go through review
             $data['status'] = Listing::STATUS_REVIEW;
        }

        // Check subscription limit if status is changing to REVIEW (or staying there but was previously something else)
        // We only care if we are moving TO a state that counts against the limit (Review/Published)
        // FROM a state that does NOT count (Draft, Sold, Expired, Rejected)
        
        $newStatus = $data['status'] ?? $listing->status;
        $isNewStatusActive = in_array($newStatus, [Listing::STATUS_PUBLISHED, Listing::STATUS_REVIEW]);
        $isOldStatusActive = in_array($listing->status, [Listing::STATUS_PUBLISHED, Listing::STATUS_REVIEW]);

        if ($isNewStatusActive && !$isOldStatusActive) {
            $canCreate = $this->premiumService->canBusinessCreateListing($business->id);
            if (!$canCreate['allowed']) {
                return ApiResponse::error($canCreate['reason'], 403);
            }
        }

        $listing->update($data);

        // Trigger Auto-Review if significant changes were made OR status moved to review
        if (($needsReview || $listing->wasChanged('status')) && $listing->status === Listing::STATUS_REVIEW) {
            AutoReviewListing::dispatch($listing->id)->delay(now()->addMinutes(rand(1, 2)));
        }

        $user = auth('api')->user();

        // Handle media update with media_ids
        // This supports: keeping existing, removing some, and adding new pre-uploaded media
        if ($request->filled('media_ids')) {
            $mediaIds = (array) $request->input('media_ids');
            $primaryIndex = $request->input('primary_media_index');

            // Get existing media for this listing
            $existingMedia = $listing->media()->get();
            $existingIds = $existingMedia->pluck('id')->toArray();

            // Delete media that was removed (not in the new media_ids list)
            $idsToDelete = array_diff($existingIds, $mediaIds);
            foreach ($existingMedia as $media) {
                if (in_array($media->id, $idsToDelete)) {
                    if ($media->path && !filter_var($media->path, FILTER_VALIDATE_URL)) {
                        Storage::disk('public')->delete($media->path);
                    }
                    if ($media->watermarked_path && !filter_var($media->watermarked_path, FILTER_VALIDATE_URL)) {
                        Storage::disk('public')->delete($media->watermarked_path);
                    }
                    $media->delete();
                }
            }

            // Link new pre-uploaded media (orphaned media with listing_id = null)
            $newMediaIds = array_diff($mediaIds, $existingIds);
            $preUploadedMedia = ListingMedia::query()
                ->whereIn('id', $newMediaIds)
                ->whereNull('listing_id')
                ->where('uploaded_by_user_id', $user?->id)
                ->get();

            foreach ($preUploadedMedia as $media) {
                $media->update([
                    'listing_id' => $listing->id,
                ]);
            }

            // Update sort_order for all media based on the order in media_ids
            foreach ($mediaIds as $i => $mediaId) {
                ListingMedia::where('id', $mediaId)
                    ->where('listing_id', $listing->id)
                    ->update([
                        'sort_order' => $i,
                        'is_primary' => $primaryIndex !== null ? ((int) $primaryIndex === (int) $i) : ($i === 0),
                    ]);
            }
        }
        // Fallback: Handle direct file uploads (backward compatibility)
        elseif ($request->hasFile('media')) {
            // Delete existing media
            $existing = $listing->media()->get();
            foreach ($existing as $media) {
                if ($media->path && !filter_var($media->path, FILTER_VALIDATE_URL)) {
                    Storage::disk('public')->delete($media->path);
                }
                if ($media->watermarked_path && !filter_var($media->watermarked_path, FILTER_VALIDATE_URL)) {
                    Storage::disk('public')->delete($media->watermarked_path);
                }
                $media->delete();
            }

            // Upload new media
            $files = (array) $request->file('media');
            $primaryIndex = $request->input('primary_media_index');
            $imageService = new ImageProcessingService();

            foreach ($files as $i => $file) {
                if (!$file) {
                    continue;
                }

                $mime = (string) $file->getClientMimeType();
                $type = str_starts_with($mime, 'video/') ? ListingMedia::TYPE_VIDEO : ListingMedia::TYPE_IMAGE;
                $contentHash = hash_file('sha256', $file->getRealPath());
                $path = $file->store('listing-media', 'public');

                $watermarkedPath = null;
                $tags = [];
                
                // Process images (compress + watermark + tag detection)
                if ($type === ListingMedia::TYPE_IMAGE && str_starts_with($mime, 'image/')) {
                    $processed = $imageService->processImage($path, $business->name);
                    $path = $processed['compressed_path'];
                    $watermarkedPath = $processed['watermarked_path'];
                    $tags = $processed['tags'] ?? [];
                }

                ListingMedia::create([
                    'listing_id' => $listing->id,
                    'uploaded_by_user_id' => $user?->id,
                    'path' => $path,
                    'watermarked_path' => $watermarkedPath,
                    'tags' => $tags,
                    'mime_type' => $mime,
                    'size_bytes' => $file->getSize(),
                    'content_hash' => $contentHash,
                    'type' => $type,
                    'sort_order' => $i,
                    'is_primary' => $primaryIndex !== null ? ((int) $primaryIndex === (int) $i) : ($i === 0),
                ]);
            }
        }

        $listing = Listing::query()
            ->whereKey($listing->getKey())
            ->with(['business', 'media'])
            ->firstOrFail();

        return ApiResponse::success(new ListingResource($listing), 'Listing updated successfully.');
    }

    public function destroy(Business $business, Listing $listing): JsonResponse
    {
        Gate::authorize('manageListings', $business);

        if ((int) $listing->business_id !== (int) $business->id) {
            return ApiResponse::notFound('Listing not found for this business.');
        }

        $existing = $listing->media()->get();
        foreach ($existing as $media) {
            if ($media->path && !filter_var($media->path, FILTER_VALIDATE_URL)) {
                Storage::disk('public')->delete($media->path);
            }
            if ($media->watermarked_path && !filter_var($media->watermarked_path, FILTER_VALIDATE_URL)) {
                Storage::disk('public')->delete($media->watermarked_path);
            }
        }

        $listing->delete();

        return ApiResponse::success(null, 'Listing deleted successfully.');
    }

    public function uploadMedia(UploadListingMediaRequest $request, Business $business): JsonResponse
    {
        Gate::authorize('manageListings', $business);

        $user = auth('api')->user();
        $uploadedMedia = [];
        $imageService = new ImageProcessingService();

        $files = (array) $request->file('media');

        foreach ($files as $i => $file) {
            if (!$file) {
                continue;
            }

            $mime = (string) $file->getClientMimeType();
            $type = str_starts_with($mime, 'video/') ? ListingMedia::TYPE_VIDEO : ListingMedia::TYPE_IMAGE;
            $contentHash = hash_file('sha256', $file->getRealPath());
            $path = $file->store('listing-media', 'public');

            $watermarkedPath = null;
            $tags = [];
            
            // Process images (compress + watermark + tag detection)
            if ($type === ListingMedia::TYPE_IMAGE && str_starts_with($mime, 'image/')) {
                $processed = $imageService->processImage($path, $business->name);
                $path = $processed['compressed_path'];
                $watermarkedPath = $processed['watermarked_path'];
                $tags = $processed['tags'] ?? [];
            }

            $media = ListingMedia::create([
                'listing_id' => null,
                'uploaded_by_user_id' => $user?->id,
                'path' => $path,
                'watermarked_path' => $watermarkedPath,
                'tags' => $tags,
                'mime_type' => $mime,
                'size_bytes' => $file->getSize(),
                'content_hash' => $contentHash,
                'type' => $type,
                'sort_order' => 0,
                'is_primary' => false,
            ]);

            $uploadedMedia[] = [
                'id' => $media->id,
                'type' => $media->type,
                'path' => $media->path,
                'watermarked_path' => $media->watermarked_path,
                'content_hash' => $media->content_hash,
                'mime_type' => $media->mime_type,
                'size_bytes' => $media->size_bytes,
            ];
        }

        return ApiResponse::success([
            'media' => $uploadedMedia,
            'message' => 'Media uploaded successfully. Use the media IDs when creating the listing.',
        ]);
    }

    public function bump(Business $business, Listing $listing): JsonResponse
    {
        Gate::authorize('manageListings', $business);

        if ((int) $listing->business_id !== (int) $business->id) {
            return ApiResponse::notFound('Listing not found for this business.');
        }

        if ($listing->status !== Listing::STATUS_PUBLISHED) {
            return ApiResponse::error('Only published listings can be bumped.', 422);
        }

        $user = auth('api')->user();
        $result = $this->premiumService->bumpListing($listing->id, $user->id, $business->id);

        if (!$result['success']) {
            return ApiResponse::error($result['message'], 403);
        }

        $listing = Listing::query()
            ->whereKey($listing->getKey())
            ->with(['business', 'media'])
            ->firstOrFail();

        return ApiResponse::success(new ListingResource($listing), 'Listing bumped successfully.');
    }

    private function normalizeTags(mixed $tags): ?array
    {
        if ($tags === null) {
            return null;
        }

        if (!is_array($tags)) {
            return null;
        }

        $out = [];
        foreach ($tags as $tag) {
            $t = trim((string) $tag);
            if ($t === '') {
                continue;
            }
            $out[] = $t;
        }

        $out = array_values(array_unique($out));

        return $out === [] ? [] : $out;
    }
}
