<?php

namespace App\Http\Controllers\Api\V1;

use App\Http\Controllers\Controller;
use App\Http\Requests\Business\RespondBusinessFeedbackRequest;
use App\Http\Resources\ApiResponse;
use App\Http\Resources\BusinessFeedbackResource;
use App\Http\Resources\BusinessFeedbackResponseResource;
use App\Models\Business;
use App\Models\BusinessFeedback;
use App\Models\BusinessFeedbackAttachment;
use App\Models\BusinessFeedbackResponse;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Gate;

class BusinessFeedbackManagementController extends Controller
{
    public function index(Request $request, Business $business): JsonResponse
    {
        Gate::authorize('update', $business);

        $perPage = (int) $request->query('per_page', 20);
        $perPage = max(1, min(100, $perPage));

        $query = BusinessFeedback::query()
            ->where('business_id', $business->id);

        if ($request->filled('moderation_status')) {
            $query->where('moderation_status', $request->query('moderation_status'));
        }

        if ($request->filled('sentiment')) {
            $query->where('sentiment', $request->query('sentiment'));
        }

        $feedback = $query
            ->with([
                'user',
                'attachments',
                'responses.attachments',
                'responses.respondedBy',
            ])
            ->orderByDesc('created_at')
            ->paginate($perPage);

        return ApiResponse::success([
            'feedback' => BusinessFeedbackResource::collection($feedback->items()),
            'pagination' => [
                'current_page' => $feedback->currentPage(),
                'per_page' => $feedback->perPage(),
                'total' => $feedback->total(),
                'last_page' => $feedback->lastPage(),
            ],
        ]);
    }

    public function respond(RespondBusinessFeedbackRequest $request, Business $business, BusinessFeedback $feedback): JsonResponse
    {
        Gate::authorize('update', $business);

        if ((int) $feedback->business_id !== (int) $business->id) {
            return ApiResponse::notFound('Feedback not found for this business.');
        }

        $user = auth('api')->user();

        $response = BusinessFeedbackResponse::create([
            'business_feedback_id' => $feedback->id,
            'business_id' => $business->id,
            'responded_by_user_id' => $user?->id,
            'message' => $request->validated()['message'],
        ]);

        if ($request->hasFile('attachments')) {
            foreach ((array) $request->file('attachments') as $file) {
                if (!$file) {
                    continue;
                }

                $path = $file->store('business-feedback-responses', 'public');

                BusinessFeedbackAttachment::create([
                    'business_feedback_response_id' => $response->id,
                    'path' => $path,
                    'mime_type' => $file->getClientMimeType(),
                    'size_bytes' => $file->getSize(),
                ]);
            }
        }

        $response = BusinessFeedbackResponse::query()
            ->whereKey($response->getKey())
            ->with(['attachments', 'respondedBy'])
            ->firstOrFail();

        return ApiResponse::created(new BusinessFeedbackResponseResource($response), 'Response added successfully.');
    }
}
