<?php

namespace App\Http\Controllers\Api\V1;

use App\Http\Controllers\Controller;
use App\Http\Requests\Business\StoreBusinessFeedbackRequest;
use App\Http\Requests\Business\UpdateBusinessFeedbackRequest;
use App\Http\Resources\ApiResponse;
use App\Http\Resources\BusinessFeedbackResource;
use App\Models\Business;
use App\Models\BusinessFeedback;
use App\Models\BusinessFeedbackAttachment;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Storage;

class BusinessFeedbackController extends Controller
{
    public function store(StoreBusinessFeedbackRequest $request, Business $business): JsonResponse
    {
        $user = auth('api')->user();

        if (!(bool) $business->is_feedback_enabled) {
            return ApiResponse::forbidden('Feedback is disabled for this business.');
        }

        $data = $request->validated();

        $feedback = BusinessFeedback::create([
            'business_id' => $business->id,
            'user_id' => $user->id,
            'message' => $data['message'],
            'sentiment' => $data['sentiment'] ?? BusinessFeedback::SENTIMENT_NEUTRAL,
            'moderation_status' => BusinessFeedback::MODERATION_PENDING,
        ]);

        if ($request->hasFile('attachments')) {
            foreach ((array) $request->file('attachments') as $file) {
                if (!$file) {
                    continue;
                }

                $path = $file->store('business-feedback', 'public');

                BusinessFeedbackAttachment::create([
                    'business_feedback_id' => $feedback->id,
                    'path' => $path,
                    'mime_type' => $file->getClientMimeType(),
                    'size_bytes' => $file->getSize(),
                ]);
            }
        }

        $feedback = BusinessFeedback::query()
            ->whereKey($feedback->getKey())
            ->with([
                'user',
                'attachments',
                'responses.attachments',
                'responses.respondedBy',
            ])
            ->firstOrFail();

        return ApiResponse::created(new BusinessFeedbackResource($feedback), 'Feedback submitted successfully.');
    }

    public function update(UpdateBusinessFeedbackRequest $request, Business $business, BusinessFeedback $feedback): JsonResponse
    {
        $user = auth('api')->user();

        if ((int) $feedback->business_id !== (int) $business->id) {
            return ApiResponse::notFound('Feedback not found for this business.');
        }

        if ((int) $feedback->user_id !== (int) $user->id) {
            return ApiResponse::forbidden('You can only update your own feedback.');
        }

        if (!(bool) $business->is_feedback_enabled) {
            return ApiResponse::forbidden('Feedback is disabled for this business.');
        }

        $data = $request->validated();

        $updates = [];
        if (array_key_exists('message', $data)) {
            $updates['message'] = $data['message'];
        }
        if (array_key_exists('sentiment', $data)) {
            $updates['sentiment'] = $data['sentiment'];
        }

        if (!empty($updates)) {
            $updates['moderation_status'] = BusinessFeedback::MODERATION_PENDING;
            $updates['reviewed_by_user_id'] = null;
            $updates['reviewed_at'] = null;
            $updates['moderation_notes'] = null;

            $feedback->update($updates);
        }

        if ($request->hasFile('attachments')) {
            $existing = $feedback->attachments()->get();
            foreach ($existing as $attachment) {
                if ($attachment->path && !filter_var($attachment->path, FILTER_VALIDATE_URL)) {
                    Storage::disk('public')->delete($attachment->path);
                }
                $attachment->delete();
            }

            foreach ((array) $request->file('attachments') as $file) {
                if (!$file) {
                    continue;
                }

                $path = $file->store('business-feedback', 'public');

                BusinessFeedbackAttachment::create([
                    'business_feedback_id' => $feedback->id,
                    'path' => $path,
                    'mime_type' => $file->getClientMimeType(),
                    'size_bytes' => $file->getSize(),
                ]);
            }
        }

        $feedback = BusinessFeedback::query()
            ->whereKey($feedback->getKey())
            ->with([
                'user',
                'attachments',
                'responses.attachments',
                'responses.respondedBy',
            ])
            ->firstOrFail();

        return ApiResponse::success(new BusinessFeedbackResource($feedback), 'Feedback updated successfully.');
    }

    public function destroy(Business $business, BusinessFeedback $feedback): JsonResponse
    {
        $user = auth('api')->user();

        if ((int) $feedback->business_id !== (int) $business->id) {
            return ApiResponse::notFound('Feedback not found for this business.');
        }

        if ((int) $feedback->user_id !== (int) $user->id) {
            return ApiResponse::forbidden('You can only delete your own feedback.');
        }

        $attachments = $feedback->attachments()->get();
        foreach ($attachments as $attachment) {
            if ($attachment->path && !filter_var($attachment->path, FILTER_VALIDATE_URL)) {
                Storage::disk('public')->delete($attachment->path);
            }
        }

        $feedback->delete();

        return ApiResponse::success(null, 'Feedback deleted successfully.');
    }
}
