<?php

namespace App\Http\Controllers\Api\V1;

use App\Http\Controllers\Controller;
use App\Http\Requests\Auth\ForgotPasswordRequest;
use App\Http\Requests\Auth\LoginRequest;
use App\Http\Requests\Auth\RegisterRequest;
use App\Http\Requests\Auth\ResetPasswordRequest;
use App\Http\Requests\Auth\SendOtpRequest;
use App\Http\Requests\Auth\VerifyOtpRequest;
use App\Http\Resources\ApiResponse;
use App\Http\Resources\AuthResource;
use App\Http\Resources\UserResource;
use App\Jobs\SendWelcomeEmail;
use App\Mail\LoginNoticeEmail;
use App\Mail\OtpCodeEmail;
use App\Models\JwtRefreshToken;
use App\Models\LoginSession;
use App\Models\OtpVerification;
use App\Models\User;
use App\Notifications\GeneralNotification;
use App\Services\PahappaSmsService;
use Carbon\Carbon;
use Illuminate\Auth\Events\PasswordReset;
use Illuminate\Auth\Events\Verified;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Password;
use Illuminate\Support\Facades\URL;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Validator;
use PHPOpenSourceSaver\JWTAuth\Facades\JWTAuth;

class AuthController extends Controller
{
    /**
     * Register a new user.
     */
    public function register(RegisterRequest $request): JsonResponse
    {
        $user = User::create([
            'full_name' => $request->full_name,
            'email' => $request->email,
            'phone' => $request->phone,
            'password' => $request->password,
            'date_of_birth' => $request->date_of_birth,
            'gender' => $request->gender,
            'status' => User::STATUS_ACTIVE,
            'role' => User::ROLE_USER,
        ]);

        // Dispatch welcome email job to queue (custom email with verification link)
        SendWelcomeEmail::dispatch($user->id);

        // Send welcome notification to database
        try {
            $user->notify(new GeneralNotification(
                'Welcome to ' . config('app.name') . '!',
                'Your account has been created successfully. Please verify your email to unlock all features.',
                ['type' => 'registration', 'user_id' => $user->id],
                false // No FCM for registration (user just created, no device token yet)
            ));
        } catch (\Throwable $e) {
            \Log::warning('Failed to send registration notification', [
                'user_id' => $user->id,
                'error' => $e->getMessage(),
            ]);
        }

        $accessToken = JWTAuth::fromUser($user);
        $refreshToken = $this->createRefreshToken($user, $request, $accessToken);

        return ApiResponse::created(
            new AuthResource($user, $accessToken, $refreshToken, config('jwt.ttl') * 60),
            'Registration successful. Please check your email to verify your account.'
        );
    }

    /**
     * Login user and return JWT tokens.
     */
    public function login(LoginRequest $request): JsonResponse
    {
        $credentials = [
            $request->loginField() => $request->login,
            'password' => $request->password,
        ];

        $user = User::where($request->loginField(), $request->login)->first();

        if (!$user) {
            return ApiResponse::unauthorized('Invalid credentials.');
        }

        if ($user->isSuspended()) {
            return ApiResponse::forbidden('Your account has been suspended. Please contact support.');
        }

        if (!$user->isActive()) {
            return ApiResponse::forbidden('Your account is not active.');
        }

        // Try standard login first
        try {
            $token = auth('api')->attempt($credentials);
        } catch (\RuntimeException $e) {
            // This happens if the password is not Bcrypt (legacy hash)
            $token = false;
        }

        // If standard login fails, check for legacy password
        if (!$token) {
            if ($user->v1_password_hash && $user->v1_salt) {
                // Legacy system used a specific sandwich concatenation logic
                $concatenatedPassword = $this->concatPasswordWithSalt($request->password, $user->v1_salt);
                
                if (password_verify($concatenatedPassword, $user->v1_password_hash)) {
                    // Success! Upgrade the user's password to the new format
                    $user->password = $request->password; // Laravel will hash it via model cast/mutator
                    $user->v1_password_hash = null;
                    $user->v1_salt = null;
                    $user->save();

                    // Now attempt standard login again with the new password
                    $token = auth('api')->attempt($credentials);
                }
            }
        }

        if (!$token) {
            return ApiResponse::unauthorized('Invalid credentials.');
        }

        $payload = JWTAuth::setToken($token)->getPayload();
        $tokenId = (string) $payload->get('jti');

        $session = null;
        try {
            $session = LoginSession::create([
                'user_id' => $user->id,
                'token_id' => $tokenId,
                'ip_address' => $request->ip(),
                'user_agent' => $request->userAgent(),
                'issued_at' => $payload->get('iat') ? Carbon::createFromTimestamp((int) $payload->get('iat')) : null,
                'expires_at' => $payload->get('exp') ? Carbon::createFromTimestamp((int) $payload->get('exp')) : null,
                'logged_in_at' => now(),
            ]);
        } catch (\Throwable $e) {
            \Log::warning('Failed to save login session', [
                'user_id' => $user->id,
                'token_id' => $tokenId,
                'error' => $e->getMessage(),
            ]);
        }

        if ($session) {
            try {
                Mail::to($user->email)->send(new LoginNoticeEmail($user, $session));
            } catch (\Throwable $e) {
                \Log::warning('Failed to send login notice email', [
                    'user_id' => $user->id,
                    'email' => $user->email,
                    'error' => $e->getMessage(),
                ]);
            }
        }

        $refreshToken = $this->createRefreshToken($user, $request, $token);

        // Send login notification to database with FCM push
        try {
            $deviceInfo = $this->parseUserAgent($request->userAgent());
            $user->notify(new GeneralNotification(
                'New Sign-In Detected',
                "You've signed in from {$deviceInfo} at " . now()->format('M d, Y h:i A') . ". If this wasn't you, please secure your account immediately.",
                ['type' => 'login', 'ip' => $request->ip(), 'device' => $deviceInfo],
                true // Send FCM push notification for login
            ));
        } catch (\Throwable $e) {
            \Log::warning('Failed to send login notification', [
                'user_id' => $user->id,
                'error' => $e->getMessage(),
            ]);
        }

        return ApiResponse::success(
            new AuthResource($user, $token, $refreshToken, config('jwt.ttl') * 60),
            'Login successful.'
        );
    }

    /**
     * Logout user and invalidate tokens.
     */
    public function logout(Request $request): JsonResponse
    {
        try {
            $token = JWTAuth::parseToken();
            $payload = $token->getPayload();
            $jti = $payload->get('jti');

            JwtRefreshToken::where('token_id', $jti)->update(['revoked_at' => now()]);

            try {
                LoginSession::where('token_id', $jti)
                    ->whereNull('logged_out_at')
                    ->update(['logged_out_at' => now()]);
            } catch (\Throwable $e) {
                \Log::warning('Failed to update login session logout time', [
                    'token_id' => $jti,
                    'error' => $e->getMessage(),
                ]);
            }

            auth('api')->logout();

            return ApiResponse::success(null, 'Successfully logged out.');
        } catch (\Exception $e) {
            return ApiResponse::error('Failed to logout.', 500);
        }
    }

    /**
     * List active login sessions (devices) for the authenticated user.
     */
    public function sessions(Request $request): JsonResponse
    {
        $user = auth('api')->user();

        $currentJti = null;
        try {
            $payload = JWTAuth::parseToken()->getPayload();
            $currentJti = (string) $payload->get('jti');
        } catch (\Throwable $e) {
            $currentJti = null;
        }

        $sessions = LoginSession::query()
            ->where('user_id', $user->id)
            ->whereNull('logged_out_at')
            ->orderByDesc('logged_in_at')
            ->get()
            ->map(function (LoginSession $session) use ($currentJti) {
                return [
                    'id' => $session->id,
                    'ip_address' => $session->ip_address,
                    'user_agent' => $session->user_agent,
                    'issued_at' => $session->issued_at?->toIso8601String(),
                    'expires_at' => $session->expires_at?->toIso8601String(),
                    'logged_in_at' => $session->logged_in_at?->toIso8601String(),
                    'is_current' => $currentJti ? $session->token_id === $currentJti : false,
                ];
            })
            ->values();

        return ApiResponse::success($sessions, 'Active sessions fetched successfully.');
    }

    /**
     * Logout (revoke) a specific session/device for the authenticated user.
     */
    public function logoutSession(Request $request, int $sessionId): JsonResponse
    {
        $user = auth('api')->user();

        $session = LoginSession::where('id', $sessionId)
            ->where('user_id', $user->id)
            ->first();

        if (!$session) {
            return ApiResponse::notFound('Session not found.');
        }

        if ($session->logged_out_at) {
            return ApiResponse::error('Session is already logged out.', 400);
        }

        $currentJti = null;
        try {
            $payload = JWTAuth::parseToken()->getPayload();
            $currentJti = (string) $payload->get('jti');
        } catch (\Throwable $e) {
            $currentJti = null;
        }

        JwtRefreshToken::where('token_id', $session->token_id)->update(['revoked_at' => now()]);

        try {
            $session->update(['logged_out_at' => now()]);
        } catch (\Throwable $e) {
            \Log::warning('Failed to update login session logout time (logoutSession)', [
                'user_id' => $user->id,
                'session_id' => $session->id,
                'token_id' => $session->token_id,
                'error' => $e->getMessage(),
            ]);
        }

        if ($currentJti && $session->token_id === $currentJti) {
            auth('api')->logout();
        }

        return ApiResponse::success(null, 'Session logged out successfully.');
    }

    /**
     * Refresh JWT token.
     */
    public function refresh(Request $request): JsonResponse
    {
        $refreshTokenString = $request->input('refresh_token');

        if (!$refreshTokenString) {
            return ApiResponse::error('Refresh token is required.', 400);
        }

        $refreshTokenHash = hash('sha256', $refreshTokenString);
        $refreshToken = JwtRefreshToken::where('refresh_token_hash', $refreshTokenHash)
            ->valid()
            ->first();

        if (!$refreshToken) {
            return ApiResponse::unauthorized('Invalid or expired refresh token.');
        }

        $user = $refreshToken->user;

        if (!$user->isActive()) {
            return ApiResponse::forbidden('Your account is not active.');
        }

        $refreshToken->revoke();

        $newAccessToken = JWTAuth::fromUser($user);
        $newRefreshToken = $this->createRefreshToken($user, $request, $newAccessToken);

        return ApiResponse::success(
            new AuthResource($user, $newAccessToken, $newRefreshToken, config('jwt.ttl') * 60),
            'Token refreshed successfully.'
        );
    }

    /**
     * Get authenticated user.
     */
    public function me(): JsonResponse
    {
        return ApiResponse::success(new UserResource(auth('api')->user()));
    }

    /**
     * Send password reset link.
     */
    public function forgotPassword(ForgotPasswordRequest $request): JsonResponse
    {
        $status = Password::sendResetLink(
            $request->only('email')
        );

        if ($status === Password::RESET_LINK_SENT) {
            return ApiResponse::success(null, 'Password reset link sent to your email.');
        }

        return ApiResponse::error('Unable to send password reset link.', 400);
    }

    /**
     * Reset password.
     */
    public function resetPassword(ResetPasswordRequest $request): JsonResponse
    {
        $status = Password::reset(
            $request->only('email', 'password', 'password_confirmation', 'token'),
            function (User $user, string $password) {
                $user->forceFill([
                    'password' => $password,
                ])->setRememberToken(Str::random(60));

                $user->save();

                JwtRefreshToken::where('user_id', $user->id)->update(['revoked_at' => now()]);

                event(new PasswordReset($user));
            }
        );

        if ($status === Password::PASSWORD_RESET) {
            return ApiResponse::success(null, 'Password has been reset successfully.');
        }

        return ApiResponse::error('Unable to reset password. Invalid or expired token.', 400);
    }

    /**
     * Resend email verification.
     */
    public function resendVerification(Request $request): JsonResponse
    {
        $user = auth('api')->user();

        if ($user->hasVerifiedEmail()) {
            return ApiResponse::error('Email already verified.', 400);
        }

        SendWelcomeEmail::dispatch($user->id);

        return ApiResponse::success(null, 'Verification email sent.');
    }

    /**
     * Verify email address.
     */
    public function verifyEmail(Request $request, int $id, string $hash)
    {
        try {
            if (!URL::hasValidSignature($request)) {
                return view('auth.email-verification-failed');
            }

            $user = User::findOrFail($id);

            if (!hash_equals((string) $hash, sha1($user->getEmailForVerification()))) {
                return view('auth.email-verification-failed');
            }

            if ($user->hasVerifiedEmail()) {
                return view('auth.email-verified', ['user' => $user]);
            }

            if ($user->markEmailAsVerified()) {
                event(new Verified($user));
            }

            return view('auth.email-verified', ['user' => $user]);
        } catch (\Exception $e) {
            return view('auth.email-verification-failed');
        }
    }

    /**
     * Send OTP for phone/email verification.
     */
    public function sendOtp(SendOtpRequest $request): JsonResponse
    {
        $identifier = $request->identifier;
        $type = $request->type;

        $user = null;
        try {
            $user = auth('api')->user();
        } catch (\Throwable $e) {
            $user = null;
        }

        if (in_array($type, [
            OtpVerification::TYPE_PHONE,
            OtpVerification::TYPE_PHONE_PRIMARY,
            OtpVerification::TYPE_PHONE_SECONDARY,
            OtpVerification::TYPE_WHATSAPP,
            OtpVerification::TYPE_WHATSAPP_PRIMARY,
            OtpVerification::TYPE_WHATSAPP_SECONDARY,
        ], true) && !$user) {
            return ApiResponse::unauthorized('Authentication is required to verify phone/WhatsApp numbers.');
        }

        if ($type === OtpVerification::TYPE_EMAIL) {
            $validator = Validator::make(['identifier' => $identifier], [
                'identifier' => ['required', 'email:rfc,dns', 'max:255'],
            ]);

            if ($validator->fails()) {
                return ApiResponse::error('Invalid email address.', 422);
            }
        }

        if (in_array($type, [
            OtpVerification::TYPE_PHONE,
            OtpVerification::TYPE_PHONE_PRIMARY,
            OtpVerification::TYPE_PHONE_SECONDARY,
            OtpVerification::TYPE_WHATSAPP,
            OtpVerification::TYPE_WHATSAPP_PRIMARY,
            OtpVerification::TYPE_WHATSAPP_SECONDARY,
        ], true)) {
            $validator = Validator::make(['identifier' => $identifier], [
                'identifier' => ['required', 'string', 'regex:/^\+?[0-9]{10,15}$/'],
            ]);

            if ($validator->fails()) {
                return ApiResponse::error('Invalid phone number. Use international format like +2567XXXXXXXX.', 422);
            }

            if ($user) {
                if (in_array($type, [OtpVerification::TYPE_PHONE, OtpVerification::TYPE_PHONE_PRIMARY], true)) {
                    if (!$user->phone || $user->phone !== $identifier) {
                        return ApiResponse::error('This phone number does not match your primary phone.', 422);
                    }
                }

                if ($type === OtpVerification::TYPE_PHONE_SECONDARY) {
                    if (!$user->secondary_phone || $user->secondary_phone !== $identifier) {
                        return ApiResponse::error('This phone number does not match your secondary phone.', 422);
                    }
                }

                if (in_array($type, [OtpVerification::TYPE_WHATSAPP, OtpVerification::TYPE_WHATSAPP_PRIMARY], true)) {
                    if (!$user->whatsapp_number || $user->whatsapp_number !== $identifier) {
                        return ApiResponse::error('This number does not match your primary WhatsApp.', 422);
                    }
                }

                if ($type === OtpVerification::TYPE_WHATSAPP_SECONDARY) {
                    if (!$user->secondary_whatsapp_number || $user->secondary_whatsapp_number !== $identifier) {
                        return ApiResponse::error('This number does not match your secondary WhatsApp.', 422);
                    }
                }
            }
        }

        if (OtpVerification::hasPending($identifier, $type)) {
            return ApiResponse::error('An OTP was recently sent. Please wait before requesting a new one.', 429);
        }

        $otpData = OtpVerification::generateFor($identifier, $type);

        if ($type === OtpVerification::TYPE_EMAIL) {
            try {
                Mail::to($identifier)->send(new OtpCodeEmail($otpData['otp'], $otpData['expires_in']));
            } catch (\Throwable $e) {
                OtpVerification::where('identifier', $identifier)
                    ->where('type', $type)
                    ->whereNull('verified_at')
                    ->delete();

                \Log::warning('Failed to send OTP email', [
                    'identifier' => $identifier,
                    'type' => $type,
                    'error' => $e->getMessage(),
                ]);

                return ApiResponse::error('Unable to send OTP email. Please try again later.', 500);
            }
        }

        if ($type === OtpVerification::TYPE_PHONE) {
            $message = 'Your ' . config('app.name') . ' verification code is ' . $otpData['otp'] . '. It expires in ' . (int) ceil($otpData['expires_in'] / 60) . ' minutes.';

            try {
                $sms = app(PahappaSmsService::class);
                $ok = $sms->send($identifier, $message);

                if (!$ok) {
                    throw new \RuntimeException('SMS gateway returned non-ok response.');
                }
            } catch (\Throwable $e) {
                OtpVerification::where('identifier', $identifier)
                    ->where('type', $type)
                    ->whereNull('verified_at')
                    ->delete();

                \Log::warning('Failed to send OTP SMS', [
                    'identifier' => $identifier,
                    'type' => $type,
                    'error' => $e->getMessage(),
                ]);

                return ApiResponse::error('Unable to send OTP SMS. Please try again later.', 500);
            }
        }

        if (in_array($type, [
            OtpVerification::TYPE_PHONE_PRIMARY,
            OtpVerification::TYPE_PHONE_SECONDARY,
            OtpVerification::TYPE_WHATSAPP,
            OtpVerification::TYPE_WHATSAPP_PRIMARY,
            OtpVerification::TYPE_WHATSAPP_SECONDARY,
        ], true)) {
            $message = 'Your ' . config('app.name') . ' verification code is ' . $otpData['otp'] . '. It expires in ' . (int) ceil($otpData['expires_in'] / 60) . ' minutes.';

            try {
                $sms = app(PahappaSmsService::class);
                $ok = $sms->send($identifier, $message);

                if (!$ok) {
                    throw new \RuntimeException('SMS gateway returned non-ok response.');
                }
            } catch (\Throwable $e) {
                OtpVerification::where('identifier', $identifier)
                    ->where('type', $type)
                    ->whereNull('verified_at')
                    ->delete();

                \Log::warning('Failed to send OTP (phone/whatsapp)', [
                    'identifier' => $identifier,
                    'type' => $type,
                    'error' => $e->getMessage(),
                ]);

                return ApiResponse::error('Unable to send OTP. Please try again later.', 500);
            }
        }

        // In production, send OTP via SMS/Email service
        // For development, return OTP in response (remove in production)
        $responseData = [
            'expires_in' => $otpData['expires_in'],
        ];

        if (config('app.debug')) {
            $responseData['otp'] = $otpData['otp'];
        }

        return ApiResponse::success($responseData, 'OTP sent successfully.');
    }

    /**
     * Verify OTP.
     */
    public function verifyOtp(VerifyOtpRequest $request): JsonResponse
    {
        $identifier = $request->identifier;
        $type = $request->type;
        $otp = $request->otp;

        $user = null;
        try {
            $user = auth('api')->user();
        } catch (\Throwable $e) {
            $user = null;
        }

        if (in_array($type, [
            OtpVerification::TYPE_PHONE,
            OtpVerification::TYPE_PHONE_PRIMARY,
            OtpVerification::TYPE_PHONE_SECONDARY,
            OtpVerification::TYPE_WHATSAPP,
            OtpVerification::TYPE_WHATSAPP_PRIMARY,
            OtpVerification::TYPE_WHATSAPP_SECONDARY,
        ], true) && !$user) {
            return ApiResponse::unauthorized('Authentication is required to verify phone/WhatsApp numbers.');
        }

        if ($user) {
            if (in_array($type, [OtpVerification::TYPE_PHONE, OtpVerification::TYPE_PHONE_PRIMARY], true) && $user->phone !== $identifier) {
                return ApiResponse::error('This OTP is not for your primary phone.', 403);
            }

            if ($type === OtpVerification::TYPE_PHONE_SECONDARY && $user->secondary_phone !== $identifier) {
                return ApiResponse::error('This OTP is not for your secondary phone.', 403);
            }

            if (in_array($type, [OtpVerification::TYPE_WHATSAPP, OtpVerification::TYPE_WHATSAPP_PRIMARY], true) && $user->whatsapp_number !== $identifier) {
                return ApiResponse::error('This OTP is not for your primary WhatsApp.', 403);
            }

            if ($type === OtpVerification::TYPE_WHATSAPP_SECONDARY && $user->secondary_whatsapp_number !== $identifier) {
                return ApiResponse::error('This OTP is not for your secondary WhatsApp.', 403);
            }
        }

        if (!OtpVerification::verify($identifier, $type, $otp)) {
            return ApiResponse::error('Invalid or expired OTP.', 400);
        }

        if ($user) {
            if ($type === OtpVerification::TYPE_EMAIL && $user->email === $identifier) {
                $user->markEmailAsVerified();
            }

            if (in_array($type, [OtpVerification::TYPE_PHONE, OtpVerification::TYPE_PHONE_PRIMARY], true)) {
                $user->markPhoneAsVerified();
            }

            if ($type === OtpVerification::TYPE_PHONE_SECONDARY) {
                $user->forceFill(['secondary_phone_verified_at' => $user->freshTimestamp()])->save();
            }

            if (in_array($type, [OtpVerification::TYPE_WHATSAPP, OtpVerification::TYPE_WHATSAPP_PRIMARY], true)) {
                $user->forceFill(['whatsapp_verified_at' => $user->freshTimestamp()])->save();
            }

            if ($type === OtpVerification::TYPE_WHATSAPP_SECONDARY) {
                $user->forceFill(['secondary_whatsapp_verified_at' => $user->freshTimestamp()])->save();
            }
        }

        return ApiResponse::success(null, 'OTP verified successfully.');
    }

    /**
     * Logout from all devices.
     */
    public function logoutAll(Request $request): JsonResponse
    {
        $user = auth('api')->user();

        JwtRefreshToken::where('user_id', $user->id)->update(['revoked_at' => now()]);

        try {
            LoginSession::where('user_id', $user->id)
                ->whereNull('logged_out_at')
                ->update(['logged_out_at' => now()]);
        } catch (\Throwable $e) {
            \Log::warning('Failed to update login sessions logout time (logoutAll)', [
                'user_id' => $user->id,
                'error' => $e->getMessage(),
            ]);
        }

        auth('api')->logout();

        return ApiResponse::success(null, 'Successfully logged out from all devices.');
    }

    /**
     * Create a refresh token for the user.
     */
    private function createRefreshToken(User $user, Request $request, string $accessToken): string
    {
        $refreshToken = Str::random(64);

        $payload = JWTAuth::setToken($accessToken)->getPayload();

        JwtRefreshToken::create([
            'user_id' => $user->id,
            'token_id' => $payload->get('jti'),
            'refresh_token_hash' => hash('sha256', $refreshToken),
            'expires_at' => now()->addMinutes(config('jwt.refresh_ttl')),
            'ip_address' => $request->ip(),
            'user_agent' => $request->userAgent(),
        ]);

        return $refreshToken;
    }

    /**
     * Parse user agent string to get a human-readable device description.
     */
    private function parseUserAgent(?string $userAgent): string
    {
        if (empty($userAgent)) {
            return 'Unknown Device';
        }

        $device = 'Unknown Device';
        $browser = '';

        // Detect device/OS
        if (preg_match('/iPhone|iPad|iPod/', $userAgent)) {
            $device = 'iOS Device';
        } elseif (preg_match('/Android/', $userAgent)) {
            $device = 'Android Device';
        } elseif (preg_match('/Windows/', $userAgent)) {
            $device = 'Windows PC';
        } elseif (preg_match('/Macintosh|Mac OS/', $userAgent)) {
            $device = 'Mac';
        } elseif (preg_match('/Linux/', $userAgent)) {
            $device = 'Linux PC';
        }

        // Detect browser
        if (preg_match('/Chrome\//', $userAgent) && !preg_match('/Edg\//', $userAgent)) {
            $browser = 'Chrome';
        } elseif (preg_match('/Safari\//', $userAgent) && !preg_match('/Chrome\//', $userAgent)) {
            $browser = 'Safari';
        } elseif (preg_match('/Firefox\//', $userAgent)) {
            $browser = 'Firefox';
        } elseif (preg_match('/Edg\//', $userAgent)) {
            $browser = 'Edge';
        }

        if ($browser) {
            return "{$device} ({$browser})";
        }

        return $device;
    }

    /**
     * Concatenate password with salt for legacy hashing.
     * Sandwich logic as seen in legacy system.
     */
    private function concatPasswordWithSalt(string $password, string $salt): string
    {
        // The legacy system used a specific $random_salt_length.
        // Based on getSalt() returning bin2hex(openssl_random_pseudo_bytes(32)),
        // the byte length was likely 32.
        $random_salt_length = 32; 
        
        if ($random_salt_length % 2 == 0) {
            $mid = $random_salt_length / 2;
        } else {
            $mid = ($random_salt_length - 1) / 2;
        }

        return substr($salt, 0, $mid - 1) . $password . substr($salt, $mid, $random_salt_length - 1);
    }

    /**
     * Get a new random salt (Legacy style).
     */
    private function getSalt(): string
    {
        $random_salt_length = 32;
        return bin2hex(openssl_random_pseudo_bytes($random_salt_length));
    }
}
