<?php

namespace App\Http\Controllers\Api\V1;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\User;
use App\Services\FcmService;
use App\Notifications\GeneralNotification;
use Illuminate\Support\Facades\Notification;

class AdminNotificationController extends Controller
{
    protected $fcmService;

    public function __construct(FcmService $fcmService)
    {
        $this->fcmService = $fcmService;
    }

    /**
     * Send notification to users.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function send(Request $request)
    {
        $request->validate([
            'title' => 'required|string',
            'body' => 'required|string',
            'user_id' => 'nullable|integer|exists:users,id',
        ]);

        $title = $request->input('title');
        $body = $request->input('body');
        $userId = $request->input('user_id');

        if ($userId) {
            // Send to specific user
            $user = User::find($userId);
            if (!$user) {
                return response()->json(['message' => 'User not found'], 404);
            }

            // 1. Save to Database (In-App)
            $user->notify(new GeneralNotification($title, $body));

            // 2. Send Push Notification
            $this->fcmService->sendToUser($user->id, $title, $body);

            return response()->json(['message' => 'Notification sent to user.']);
        } else {
            // Send to ALL users
            // Process in chunks to avoid memory issues
            User::chunk(100, function ($users) use ($title, $body) {
                foreach ($users as $user) {
                    // 1. Save to Database (In-App)
                    // Queue this to avoid timeout
                    $user->notify(new GeneralNotification($title, $body));

                    // 2. Send Push Notification
                    // Ensure FcmService can handle individual sends efficiently or we might need a topic/multicast later
                    // For now, loop send is okay for smaller user bases, but typically we'd use topics for "All Users"
                    $this->fcmService->sendToUser($user->id, $title, $body);
                }
            });

            return response()->json(['message' => 'Notification broadcast started for all users.']);
        }
    }
}
