<?php

namespace App\Http\Controllers\Api\V1;

use App\Http\Controllers\Controller;
use App\Http\Requests\Admin\ListUsersRequest;
use App\Http\Requests\Admin\UpdateUserRoleRequest;
use App\Http\Requests\Admin\UpdateUserStatusRequest;
use App\Http\Resources\ApiResponse;
use App\Http\Resources\UserCollection;
use App\Http\Resources\UserResource;
use App\Models\JwtRefreshToken;
use App\Models\User;
use Illuminate\Http\JsonResponse;

class AdminController extends Controller
{
    /**
     * List all users with filtering and pagination.
     */
    public function listUsers(ListUsersRequest $request): JsonResponse
    {
        $query = User::query();

        if ($request->boolean('include_deleted')) {
            $query->withTrashed();
        }

        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('full_name', 'like', "%{$search}%")
                    ->orWhere('email', 'like', "%{$search}%")
                    ->orWhere('phone', 'like', "%{$search}%");
            });
        }

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('role')) {
            $query->where('role', $request->role);
        }

        $sortBy = $request->input('sort_by', 'created_at');
        $sortOrder = $request->input('sort_order', 'desc');
        $query->orderBy($sortBy, $sortOrder);

        $perPage = $request->input('per_page', 15);
        $users = $query->paginate($perPage);

        return ApiResponse::success(new UserCollection($users));
    }

    /**
     * Get a specific user.
     */
    public function showUser(int $id): JsonResponse
    {
        $user = User::withTrashed()->findOrFail($id);

        return ApiResponse::success(new UserResource($user));
    }

    /**
     * Update user status (activate/suspend).
     */
    public function updateUserStatus(UpdateUserStatusRequest $request, int $id): JsonResponse
    {
        $user = User::findOrFail($id);
        $admin = auth('api')->user();

        if ($user->id == $admin->id) {
            return ApiResponse::error('You cannot change your own status.', 400);
        }

        $oldStatus = $user->status;
        $user->update(['status' => $request->status]);

        if ($request->status === User::STATUS_SUSPENDED) {
            JwtRefreshToken::where('user_id', $user->id)->update(['revoked_at' => now()]);
        }

        return ApiResponse::success(
            new UserResource($user->fresh()),
            "User status changed from {$oldStatus} to {$request->status}."
        );
    }

    /**
     * Update user role.
     */
    public function updateUserRole(UpdateUserRoleRequest $request, int $id): JsonResponse
    {
        $user = User::findOrFail($id);
        $admin = auth('api')->user();

        if ($user->id == $admin->id) {
            return ApiResponse::error('You cannot change your own role.', 400);
        }

        $oldRole = $user->role;
        $user->update(['role' => $request->role]);

        return ApiResponse::success(
            new UserResource($user->fresh()),
            "User role changed from {$oldRole} to {$request->role}."
        );
    }

    /**
     * Force delete a user (permanent).
     */
    public function forceDeleteUser(int $id): JsonResponse
    {
        $user = User::withTrashed()->findOrFail($id);
        $admin = auth('api')->user();

        if ($user->id == $admin->id) {
            return ApiResponse::error('You cannot delete your own account.', 400);
        }

        JwtRefreshToken::where('user_id', $user->id)->delete();

        $user->forceDelete();

        return ApiResponse::success(null, 'User permanently deleted.');
    }

    /**
     * Restore a soft-deleted user.
     */
    public function restoreUser(int $id): JsonResponse
    {
        $user = User::withTrashed()->findOrFail($id);

        if (!$user->trashed()) {
            return ApiResponse::error('User is not deleted.', 400);
        }

        $user->restore();
        $user->update(['status' => User::STATUS_ACTIVE]);

        return ApiResponse::success(
            new UserResource($user->fresh()),
            'User restored successfully.'
        );
    }

    /**
     * Get user statistics.
     */
    public function statistics(): JsonResponse
    {
        $stats = [
            'total_users' => User::count(),
            'active_users' => User::where('status', User::STATUS_ACTIVE)->count(),
            'suspended_users' => User::where('status', User::STATUS_SUSPENDED)->count(),
            'deleted_users' => User::onlyTrashed()->count(),
            'verified_emails' => User::whereNotNull('email_verified_at')->count(),
            'verified_phones' => User::whereNotNull('phone_verified_at')->count(),
            'by_role' => [
                'users' => User::where('role', User::ROLE_USER)->count(),
                'sellers' => User::where('role', User::ROLE_SELLER)->count(),
                'admins' => User::where('role', User::ROLE_ADMIN)->count(),
            ],
            'registrations_today' => User::whereDate('created_at', today())->count(),
            'registrations_this_week' => User::whereBetween('created_at', [now()->startOfWeek(), now()->endOfWeek()])->count(),
            'registrations_this_month' => User::whereMonth('created_at', now()->month)->whereYear('created_at', now()->year)->count(),
        ];

        return ApiResponse::success($stats);
    }
}
