<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Http\RedirectResponse;
use Illuminate\View\View;
use Illuminate\Support\Facades\Artisan;
use App\Services\FirebaseService;

class SettingsController extends Controller
{
    protected FirebaseService $firebaseService;

    public function __construct(FirebaseService $firebaseService)
    {
        $this->firebaseService = $firebaseService;
    }

    /**
     * Display system settings.
     */
    public function index(): View
    {
        // Read env for display
        $env = [
            // App
            'APP_NAME' => env('APP_NAME'),
            'APP_ENV' => env('APP_ENV'),
            'APP_DEBUG' => env('APP_DEBUG'),
            'APP_URL' => env('APP_URL'),
            'APP_TIMEZONE' => env('APP_TIMEZONE', 'UTC'),
            
            // Database
            'DB_CONNECTION' => env('DB_CONNECTION'),
            'DB_HOST' => env('DB_HOST'),
            'DB_PORT' => env('DB_PORT'),
            'DB_DATABASE' => env('DB_DATABASE'),
            'DB_USERNAME' => env('DB_USERNAME'),
            'DB_PASSWORD' => env('DB_PASSWORD'),

            // Mail
            'MAIL_MAILER' => env('MAIL_MAILER'),
            'MAIL_HOST' => env('MAIL_HOST'),
            'MAIL_PORT' => env('MAIL_PORT'),
            'MAIL_USERNAME' => env('MAIL_USERNAME'),
            'MAIL_PASSWORD' => env('MAIL_PASSWORD'),
            'MAIL_ENCRYPTION' => env('MAIL_ENCRYPTION'),
            'MAIL_FROM_ADDRESS' => env('MAIL_FROM_ADDRESS'),
            'MAIL_FROM_NAME' => env('MAIL_FROM_NAME'),

            // Filesystem (AWS)
            'FILESYSTEM_DISK' => env('FILESYSTEM_DISK'),
            'AWS_ACCESS_KEY_ID' => env('AWS_ACCESS_KEY_ID'),
            'AWS_SECRET_ACCESS_KEY' => env('AWS_SECRET_ACCESS_KEY'),
            'AWS_DEFAULT_REGION' => env('AWS_DEFAULT_REGION'),
            'AWS_BUCKET' => env('AWS_BUCKET'),
            'AWS_USE_PATH_STYLE_ENDPOINT' => env('AWS_USE_PATH_STYLE_ENDPOINT'),

            // System
            'QUEUE_CONNECTION' => env('QUEUE_CONNECTION'),
            'CACHE_STORE' => env('CACHE_STORE'),
            'SESSION_DRIVER' => env('SESSION_DRIVER'),
            'SESSION_LIFETIME' => env('SESSION_LIFETIME'),

            // Security & Logging
            'JWT_SECRET' => env('JWT_SECRET'),
            'LOG_CHANNEL' => env('LOG_CHANNEL'),
            'LOG_LEVEL' => env('LOG_LEVEL'),

            // Integrations
            'FIREBASE_PROJECT_ID' => env('FIREBASE_PROJECT_ID'),
            
            // SMS (EgoSMS)
            'EGOSMS_USERNAME' => env('EGOSMS_USERNAME'),
            'EGOSMS_PASSWORD' => env('EGOSMS_PASSWORD'),
            'EGOSMS_SENDER' => env('EGOSMS_SENDER'),
            
            // WhatsApp
            'WHATSAPP_ACCESS_TOKEN' => env('WHATSAPP_ACCESS_TOKEN'),
            'WHATSAPP_PHONE_NUMBER_ID' => env('WHATSAPP_PHONE_NUMBER_ID'),
            
            // AI & Vision
            'OPENAI_API_KEY' => env('OPENAI_API_KEY'),
            'OPENAI_MODEL' => env('OPENAI_MODEL'),
            'GOOGLE_VISION_API_KEY' => env('GOOGLE_VISION_API_KEY'),
        ];

        // Fetch Company Info from Firebase
        $firebaseData = $this->firebaseService->getCompanyInfo();

        return view('admin.settings.index', compact('env', 'firebaseData'));
    }

    /**
     * Update Company Info (Firebase).
     */
    public function updateCompanyInfo(Request $request): RedirectResponse
    {
        $data = $request->except(['_token', '_method']);

        // Recursively convert array looking keys if necessary, or just rely on dot notation in form names
        // Ideally the form sends nested arrays: android[version], etc.
        
        // Structure fixup if needed for boolean strings
        if (isset($data['android']['is_forced'])) $data['android']['is_forced'] = filter_var($data['android']['is_forced'], FILTER_VALIDATE_BOOLEAN);
        if (isset($data['ios']['is_forced'])) $data['ios']['is_forced'] = filter_var($data['ios']['is_forced'], FILTER_VALIDATE_BOOLEAN);
        if (isset($data['maintenance']['is_active'])) $data['maintenance']['is_active'] = filter_var($data['maintenance']['is_active'], FILTER_VALIDATE_BOOLEAN);

        $success = $this->firebaseService->updateCompanyInfo($data);

        if ($success) {
            return back()->with('success', 'Company information updated in Firebase.');
        }

        return back()->with('error', 'Failed to update company information in Firebase.');
    }

    /**
     * Update system settings (.env).
     */
    public function update(Request $request): RedirectResponse
    {
        // Validation rules can be expansive based on what we allow editing.
        // For now, we allow generic string inputs for common keys.
        $data = $request->except(['_token', '_method']);
        
        $this->updateEnvironmentFile($data);

        // Clear config cache to ensure changes take effect
        Artisan::call('config:clear');
        Artisan::call('cache:clear');

        return back()->with('success', 'System settings updated successfully.');
    }

    /**
     * Update .env file safely.
     */
    protected function updateEnvironmentFile(array $data)
    {
        $path = base_path('.env');

        if (!file_exists($path)) {
            return;
        }

        $file_content = file_get_contents($path);
        $lines = explode("\n", $file_content);
        $new_content = [];
        $keys_processed = [];

        // Parse existing lines
        foreach ($lines as $line) {
            $line_trimmed = trim($line);
            if (empty($line_trimmed) || str_starts_with($line_trimmed, '#')) {
                $new_content[] = $line;
                continue;
            }

            $parts = explode('=', $line, 2);
            if (count($parts) < 2) {
                $new_content[] = $line;
                continue;
            }

            $key = trim($parts[0]);
            
            // If this key is in our update data, use the new value
            if (array_key_exists($key, $data)) {
                $value = $data[$key];
                // Quote value if it contains spaces or special chars, but simplest is to always quote strings
                // or just keep it simple. Laravel env handling usually tolerates quotes.
                if (strpos($value, ' ') !== false || strpos($value, '"') !== false) {
                     $value = '"' . str_replace('"', '\"', $value) . '"';
                }
                $new_content[] = "{$key}={$value}";
                $keys_processed[] = $key;
            } else {
                $new_content[] = $line;
            }
        }

        // Add new keys that weren't in the file
        foreach ($data as $key => $value) {
            if (!in_array($key, $keys_processed)) {
                if (strpos($value, ' ') !== false || strpos($value, '"') !== false) {
                     $value = '"' . str_replace('"', '\"', $value) . '"';
                }
                $new_content[] = "{$key}={$value}";
            }
        }

        file_put_contents($path, implode("\n", $new_content));
    }
}
