<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Listing;
use App\Notifications\AdPublishedNotification;
use App\Notifications\AdRejectedNotification;
use Illuminate\Http\Request;
use Illuminate\Http\RedirectResponse;
use Illuminate\View\View;

class ListingController extends Controller
{
    /**
     * Display a listing of the ads.
     */
    public function index(Request $request): View
    {
        $query = Listing::with(['business', 'createdBy']);

        // Search
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('title', 'like', "%{$search}%")
                    ->orWhere('description', 'like', "%{$search}%");
            });
        }

        // Filter by status
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        // Filter by category
        if ($request->filled('category')) {
            $query->where('category', $request->category);
        }

        $listings = $query->latest()->paginate(20)->withQueryString();

        return view('admin.listings.index', compact('listings'));
    }

    /**
     * Display the specified listing.
     */
    public function show(Listing $listing): View
    {
        $listing->load(['business', 'createdBy', 'media']);
        return view('admin.listings.show', compact('listing'));
    }

    /**
     * Update listing status.
     */
    public function updateStatus(Request $request, Listing $listing): RedirectResponse
    {
        $request->validate([
            'status' => 'required|in:' . implode(',', [
                Listing::STATUS_PUBLISHED, 
                Listing::STATUS_REVIEW, 
                Listing::STATUS_REJECTED, 
                Listing::STATUS_DRAFT
            ]),
            'reason' => 'required_if:status,' . Listing::STATUS_REJECTED . '|nullable|string|max:1000',
        ]);

        $updateData = ['status' => $request->status];
        
        if ($request->status === Listing::STATUS_REJECTED) {
            $updateData['decline_reason'] = $request->reason;
        }

        $listing->update($updateData);

        // Notify the owner
        if ($request->status === Listing::STATUS_PUBLISHED) {
            $listing->createdBy->notify(new AdPublishedNotification($listing));
        } elseif ($request->status === Listing::STATUS_REJECTED) {
            $listing->createdBy->notify(new AdRejectedNotification($listing));
        }

        return back()->with('success', "Listing status updated to {$request->status}.");
    }

    /**
     * Bulk update listing status.
     */
    public function bulkStatus(Request $request): RedirectResponse
    {
        $request->validate([
            'listings' => 'required|array',
            'listings.*' => 'exists:listings,id',
            'status' => 'required|in:' . implode(',', [
                Listing::STATUS_PUBLISHED, 
                Listing::STATUS_REVIEW, 
                Listing::STATUS_REJECTED, 
                Listing::STATUS_DRAFT
            ]),
        ]);

        $listings = Listing::whereIn('id', $request->listings)->get();

        foreach ($listings as $listing) {
            $listing->update(['status' => $request->status]);

            // Notify the owner
            if ($request->status === Listing::STATUS_PUBLISHED) {
                $listing->createdBy->notify(new AdPublishedNotification($listing));
            } elseif ($request->status === Listing::STATUS_REJECTED) {
                $listing->createdBy->notify(new AdRejectedNotification($listing));
            }
        }

        $count = $listings->count();
        return back()->with('success', "{$count} listings updated to {$request->status}.");
    }

    /**
     * Remove the specified listing from storage.
     */
    public function destroy(Listing $listing): RedirectResponse
    {
        $listing->delete();
        return redirect()->route('admin.listings.index')->with('success', 'Listing deleted successfully.');
    }
}
