# API Documentation - Yilick Classified Platform

**Base URL:** `/api/v1`  
**Authentication:** JWT Bearer Token (where required)

---

## Table of Contents

1. [Authentication](#authentication)
2. [User Management](#user-management)
3. [Business Management](#business-management)
4. [Listings](#listings)
5. [Analytics & Tracking](#analytics--tracking)
6. [Feedback](#feedback)
7. [Admin](#admin)
8. [Error Responses](#error-responses)

---

## Authentication

### Register User
**POST** `/auth/register`

**Description:** Create a new user account.

**Authentication:** None (Public)

**Request Body:**
```json
{
  "first_name": "John",              // Required, string, max:255
  "last_name": "Doe",                // Required, string, max:255
  "email": "john@example.com",       // Required, email, unique
  "phone_number": "+256700123456",   // Required, string, E164 format, unique
  "password": "SecurePass123!",      // Required, min:8, confirmed
  "password_confirmation": "SecurePass123!",
  "gender": "male",                  // Optional, enum: male|female|other
  "date_of_birth": "1990-01-15",     // Optional, date, YYYY-MM-DD
  "whatsapp_number": "+256700123456" // Optional, string, E164 format
}
```

**Success Response (201):**
```json
{
  "success": true,
  "message": "Registration successful. Please verify your email.",
  "data": {
    "user": {
      "id": 1,
      "first_name": "John",
      "last_name": "Doe",
      "email": "john@example.com",
      "phone_number": "+256700123456",
      "role": "buyer",
      "status": "active",
      "email_verified_at": null,
      "created_at": "2025-12-21T06:00:00Z"
    },
    "access_token": "eyJ0eXAiOiJKV1QiLCJhbGc...",
    "refresh_token": "abc123def456...",
    "token_type": "bearer",
    "expires_in": 3600
  }
}
```

**Error Response (422):**
```json
{
  "success": false,
  "message": "Validation failed",
  "errors": {
    "email": ["The email has already been taken."],
    "password": ["The password must be at least 8 characters."]
  }
}
```

---

### Login
**POST** `/auth/login`

**Description:** Authenticate user and receive JWT tokens.

**Authentication:** None (Public)

**Request Body:**
```json
{
  "email": "john@example.com",    // Required, email
  "password": "SecurePass123!"    // Required, string
}
```

**Success Response (200):**
```json
{
  "success": true,
  "message": "Login successful",
  "data": {
    "user": {
      "id": 1,
      "first_name": "John",
      "last_name": "Doe",
      "email": "john@example.com",
      "role": "seller",
      "status": "active"
    },
    "access_token": "eyJ0eXAiOiJKV1QiLCJhbGc...",
    "refresh_token": "abc123def456...",
    "token_type": "bearer",
    "expires_in": 3600
  }
}
```

**Error Response (401):**
```json
{
  "success": false,
  "message": "Invalid credentials"
}
```

---

### Refresh Token
**POST** `/auth/refresh`

**Description:** Get new access token using refresh token.

**Authentication:** None (Public)

**Request Body:**
```json
{
  "refresh_token": "abc123def456..."  // Required, string
}
```

**Success Response (200):**
```json
{
  "success": true,
  "data": {
    "access_token": "eyJ0eXAiOiJKV1QiLCJhbGc...",
    "refresh_token": "xyz789ghi012...",
    "token_type": "bearer",
    "expires_in": 3600
  }
}
```

**Error Response (401):**
```json
{
  "success": false,
  "message": "Invalid or expired refresh token"
}
```

---

### Get Current User
**GET** `/auth/me`

**Description:** Get authenticated user's profile.

**Authentication:** Required (Bearer Token)

**Success Response (200):**
```json
{
  "success": true,
  "data": {
    "id": 1,
    "first_name": "John",
    "last_name": "Doe",
    "email": "john@example.com",
    "phone_number": "+256700123456",
    "whatsapp_number": "+256700123456",
    "role": "seller",
    "status": "active",
    "gender": "male",
    "date_of_birth": "1990-01-15",
    "profile_photo_url": "https://example.com/storage/profile-photos/user-1.jpg",
    "email_verified_at": "2025-12-21T06:00:00Z",
    "created_at": "2025-12-21T06:00:00Z"
  }
}
```

---

### Logout
**POST** `/auth/logout`

**Description:** Logout from current device (invalidate access token).

**Authentication:** Required (Bearer Token)

**Success Response (200):**
```json
{
  "success": true,
  "message": "Logged out successfully"
}
```

---

### Logout All Devices
**POST** `/auth/logout-all`

**Description:** Logout from all devices (revoke all refresh tokens).

**Authentication:** Required (Bearer Token)

**Success Response (200):**
```json
{
  "success": true,
  "message": "Logged out from all devices successfully"
}
```

---

## User Management

### Update Profile
**PUT** `/user/profile`

**Description:** Update authenticated user's profile.

**Authentication:** Required (Bearer Token)

**Request Body (multipart/form-data):**
```json
{
  "first_name": "John",              // Optional, string, max:255
  "last_name": "Doe",                // Optional, string, max:255
  "phone_number": "+256700123456",   // Optional, string, E164 format, unique
  "whatsapp_number": "+256700123456",// Optional, string, E164 format
  "gender": "male",                  // Optional, enum: male|female|other
  "date_of_birth": "1990-01-15",     // Optional, date, YYYY-MM-DD
  "profile_photo": "<file>",         // Optional, image, max:5MB, jpeg/png/jpg
  "bio": "Software developer..."     // Optional, string, max:500
}
```

**Success Response (200):**
```json
{
  "success": true,
  "message": "Profile updated successfully",
  "data": {
    "id": 1,
    "first_name": "John",
    "last_name": "Doe",
    "profile_photo_url": "https://example.com/storage/profile-photos/user-1.jpg"
  }
}
```

---

### Change Password
**PUT** `/user/change-password`

**Description:** Change user's password.

**Authentication:** Required (Bearer Token)

**Request Body:**
```json
{
  "current_password": "OldPass123!",     // Required, string
  "password": "NewPass123!",             // Required, min:8, confirmed
  "password_confirmation": "NewPass123!" // Required
}
```

**Success Response (200):**
```json
{
  "success": true,
  "message": "Password changed successfully"
}
```

**Error Response (422):**
```json
{
  "success": false,
  "message": "Current password is incorrect"
}
```

---

## Business Management

### Create Business
**POST** `/businesses`

**Description:** Create a new business profile (promotes user to seller role).

**Authentication:** Required (Bearer Token)

**Request Body (multipart/form-data):**
```json
{
  "name": "John's Electronics",          // Required, string, max:255, unique
  "description": "Best electronics...",  // Optional, string
  "category": "Electronics",             // Optional, string
  "email": "contact@johns.com",          // Optional, email
  "phone_number": "+256700123456",       // Optional, string, E164 format
  "whatsapp_number": "+256700123456",    // Optional, string, E164 format
  "website": "https://johns.com",        // Optional, url
  "address": "123 Main St, Kampala",     // Optional, string
  "city": "Kampala",                     // Optional, string
  "country_code": "UG",                  // Optional, string, 2 chars
  "latitude": 0.3476,                    // Optional, numeric
  "longitude": 32.5825,                  // Optional, numeric
  "logo": "<file>",                      // Optional, image, max:5MB
  "cover_photo": "<file>",               // Optional, image, max:10MB
  "is_feedback_enabled": true            // Optional, boolean, default:true
}
```

**Success Response (201):**
```json
{
  "success": true,
  "message": "Business created successfully",
  "data": {
    "id": 1,
    "name": "John's Electronics",
    "slug": "johns-electronics",
    "description": "Best electronics in town",
    "category": "Electronics",
    "owner_id": 1,
    "verification_status": "pending",
    "logo_url": "https://example.com/storage/business-logos/1.jpg",
    "cover_photo_url": "https://example.com/storage/business-covers/1.jpg",
    "followers_count": 0,
    "created_at": "2025-12-21T06:00:00Z"
  }
}
```

---

### Get Business by ID
**GET** `/businesses/{business_id}`

**Description:** Get business details (owner/members only).

**Authentication:** Required (Bearer Token)

**URL Parameters:**
- `business_id` - Integer, required

**Success Response (200):**
```json
{
  "success": true,
  "data": {
    "id": 1,
    "name": "John's Electronics",
    "slug": "johns-electronics",
    "description": "Best electronics in town",
    "owner": {
      "id": 1,
      "first_name": "John",
      "last_name": "Doe"
    },
    "contact": {
      "email": "contact@johns.com",
      "phone_number": "+256700123456",
      "whatsapp_number": "+256700123456"
    },
    "verification_status": "verified",
    "followers_count": 150,
    "is_following": false
  }
}
```

---

### Get Business by Slug (Public)
**GET** `/businesses/{slug}`

**Description:** Get public business profile by slug.

**Authentication:** None (Public)

**URL Parameters:**
- `slug` - String, required (e.g., "johns-electronics")

**Success Response (200):**
```json
{
  "success": true,
  "data": {
    "id": 1,
    "name": "John's Electronics",
    "slug": "johns-electronics",
    "description": "Best electronics in town",
    "category": "Electronics",
    "logo_url": "https://example.com/storage/business-logos/1.jpg",
    "cover_photo_url": "https://example.com/storage/business-covers/1.jpg",
    "followers_count": 150,
    "verification_status": "verified",
    "contact": {
      "email": "contact@johns.com",
      "phone_number": "+256700123456",
      "whatsapp_number": "+256700123456",
      "website": "https://johns.com"
    },
    "location": {
      "address": "123 Main St, Kampala",
      "city": "Kampala",
      "country_code": "UG"
    }
  }
}
```

---

### Update Business
**PUT** `/businesses/{business_id}`

**Description:** Update business profile (owner/manager only).

**Authentication:** Required (Bearer Token)

**URL Parameters:**
- `business_id` - Integer, required

**Request Body:** Same as Create Business (all fields optional)

**Success Response (200):**
```json
{
  "success": true,
  "message": "Business updated successfully",
  "data": { /* business object */ }
}
```

---

### Follow Business
**POST** `/businesses/{business_id}/follow`

**Description:** Follow a business to get updates.

**Authentication:** Required (Bearer Token)

**URL Parameters:**
- `business_id` - Integer, required

**Success Response (200):**
```json
{
  "success": true,
  "message": "Now following John's Electronics"
}
```

---

### Unfollow Business
**DELETE** `/businesses/{business_id}/follow`

**Description:** Unfollow a business.

**Authentication:** Required (Bearer Token)

**URL Parameters:**
- `business_id` - Integer, required

**Success Response (200):**
```json
{
  "success": true,
  "message": "Unfollowed John's Electronics"
}
```

---

## Listings

### Browse Listings (Public)
**GET** `/listings`

**Description:** Browse all published listings with search, filters, and pagination.

**Authentication:** None (Public)

**Query Parameters:**
```
?search=red dress           // Optional, string - search in title/description
&category=Fashion           // Optional, string - filter by category
&sub_category=Dresses       // Optional, string - filter by sub-category
&min_price=1000            // Optional, numeric - minimum price
&max_price=50000           // Optional, numeric - maximum price
&location_city=Kampala     // Optional, string - filter by city
&location_district=Central // Optional, string - filter by district
&sort=price_asc            // Optional, enum: price_asc|price_desc|newest|oldest|bumped
&per_page=20               // Optional, integer, 1-100, default:20
&page=1                    // Optional, integer, default:1
```

**Success Response (200):**
```json
{
  "success": true,
  "data": {
    "listings": [
      {
        "id": 1,
        "title": "Red Evening Dress - Size M",
        "slug": "red-evening-dress-size-m-1",
        "description": "Beautiful red dress...",
        "price": {
          "amount": "25000.00",
          "currency": "UGX",
          "formatted": "UGX 25,000"
        },
        "category": "Fashion",
        "sub_category": "Dresses",
        "status": "published",
        "inventory": {
          "quantity": 1,
          "sku": "RD-001",
          "allow_pos": false
        },
        "location": {
          "city": "Kampala",
          "district": "Central",
          "country_code": "UG"
        },
        "stats": {
          "views_count": 289,
          "unique_views_count": 245,
          "impressions_count": 5420,
          "clicks_count": 342,
          "phone_reveals_count": 37,
          "whatsapp_clicks_count": 28,
          "shares_count": 15
        },
        "business": {
          "id": 1,
          "name": "John's Fashion",
          "slug": "johns-fashion",
          "logo_url": "https://example.com/storage/logos/1.jpg"
        },
        "media": [
          {
            "id": 1,
            "type": "image",
            "path": "listing-media/dress-1.jpg",
            "watermarked_path": "listing-media/dress-1-watermarked.jpg",
            "url": "https://example.com/storage/listing-media/dress-1.jpg",
            "watermarked_url": "https://example.com/storage/listing-media/dress-1-watermarked.jpg",
            "is_primary": true,
            "tags": [
              {"label": "red dress", "confidence": 0.95},
              {"label": "evening wear", "confidence": 0.88}
            ]
          }
        ],
        "tags": ["red dress", "evening wear", "size M"],
        "created_at": "2025-12-21T06:00:00Z",
        "published_at": "2025-12-21T06:00:00Z",
        "bumped_at": "2025-12-21T08:00:00Z"
      }
    ],
    "pagination": {
      "current_page": 1,
      "per_page": 20,
      "total": 150,
      "last_page": 8
    }
  }
}
```

---

### Get Listing by Slug (Public)
**GET** `/listings/{slug}`

**Description:** Get full listing details by slug. **Automatically tracks view.**

**Authentication:** None (Public)

**URL Parameters:**
- `slug` - String, required (e.g., "red-evening-dress-size-m-1")

**Success Response (200):**
```json
{
  "success": true,
  "data": {
    "listing": { /* full listing object */ },
    "related_listings": [ /* array of related listings */ ]
  }
}
```

---

### Create Listing
**POST** `/businesses/{business_id}/listings`

**Description:** Create a new listing for a business.

**Authentication:** Required (Bearer Token)

**URL Parameters:**
- `business_id` - Integer, required

**Request Body (multipart/form-data):**
```json
{
  "title": "Red Evening Dress",           // Required, string, max:255
  "description": "Beautiful dress...",    // Required, string
  "category": "Fashion",                  // Optional, string
  "sub_category": "Dresses",              // Optional, string
  "sub_sub_category": "Evening Dresses",  // Optional, string
  "price_amount": 25000,                  // Optional, numeric, min:0
  "purchase_price_amount": 15000,         // Optional, numeric, min:0
  "currency": "UGX",                      // Optional, string, default:UGX
  "status": "published",                  // Optional, enum: draft|published|review, default:draft
  "quantity": 1,                          // Optional, integer, min:0, default:1
  "sku": "RD-001",                        // Optional, string, unique
  "barcode": "123456789",                 // Optional, string
  "allow_pos": false,                     // Optional, boolean, default:false
  "location_city": "Kampala",             // Optional, string
  "location_district": "Central",         // Optional, string
  "location_country_code": "UG",          // Optional, string, 2 chars
  "latitude": 0.3476,                     // Optional, numeric
  "longitude": 32.5825,                   // Optional, numeric
  "attributes": {                         // Optional, object
    "color": "red",
    "size": "M",
    "brand": "Zara"
  },
  "custom_fields": {                      // Optional, object
    "material": "silk",
    "care_instructions": "Dry clean only"
  },
  "tags": ["red dress", "evening wear"],  // Optional, array of strings
  "meta_title": "Red Evening Dress...",   // Optional, string, max:60
  "meta_description": "Buy red dress...", // Optional, string, max:160
  "meta_keywords": "red,dress,evening",   // Optional, string
  "canonical_url": "https://...",         // Optional, url
  "expires_at": "2025-12-31T23:59:59Z",   // Optional, datetime
  
  // Media options (choose one):
  // Option 1: Pre-uploaded media IDs
  "media_ids": [1, 2, 3],                 // Optional, array of integers
  "primary_media_index": 0,               // Optional, integer, default:0
  
  // Option 2: Direct file upload
  "media": ["<file1>", "<file2>"],        // Optional, array of files, max:10, each max:10MB
  "primary_media_index": 0                // Optional, integer, default:0
}
```

**Success Response (201):**
```json
{
  "success": true,
  "message": "Listing created successfully",
  "data": { /* full listing object */ }
}
```

**Error Response (422):**
```json
{
  "success": false,
  "message": "Validation failed",
  "errors": {
    "title": ["The title field is required."],
    "price_amount": ["The price amount must be at least 0."]
  }
}
```

---

### Pre-Upload Listing Media
**POST** `/businesses/{business_id}/listings/media/upload`

**Description:** Pre-upload media files before creating listing. Returns media IDs to use in listing creation.

**Authentication:** Required (Bearer Token)

**URL Parameters:**
- `business_id` - Integer, required

**Request Body (multipart/form-data):**
```json
{
  "media": ["<file1>", "<file2>", "<file3>"]  // Required, array of files, max:10 files, each max:10MB, jpeg/png/jpg/mp4
}
```

**Success Response (200):**
```json
{
  "success": true,
  "message": "Media uploaded successfully",
  "data": {
    "media": [
      {
        "id": 1,
        "type": "image",
        "path": "listing-media/temp-1.jpg",
        "watermarked_path": "listing-media/temp-1-watermarked.jpg",
        "url": "https://example.com/storage/listing-media/temp-1.jpg",
        "size_bytes": 245678,
        "tags": [
          {"label": "red dress", "confidence": 0.95}
        ]
      },
      {
        "id": 2,
        "type": "image",
        "path": "listing-media/temp-2.jpg",
        "watermarked_path": "listing-media/temp-2-watermarked.jpg",
        "url": "https://example.com/storage/listing-media/temp-2.jpg",
        "size_bytes": 198432,
        "tags": [
          {"label": "evening wear", "confidence": 0.88}
        ]
      }
    ]
  }
}
```

**Usage:**
1. Upload media files using this endpoint
2. Get media IDs from response
3. Use media IDs in listing creation: `"media_ids": [1, 2]`

---

### Update Listing
**PUT** `/businesses/{business_id}/listings/{listing_id}`

**Description:** Update existing listing.

**Authentication:** Required (Bearer Token)

**URL Parameters:**
- `business_id` - Integer, required
- `listing_id` - Integer, required

**Request Body:** Same as Create Listing (all fields optional)

**Success Response (200):**
```json
{
  "success": true,
  "message": "Listing updated successfully",
  "data": { /* full listing object */ }
}
```

---

### Delete Listing
**DELETE** `/businesses/{business_id}/listings/{listing_id}`

**Description:** Soft delete a listing.

**Authentication:** Required (Bearer Token)

**URL Parameters:**
- `business_id` - Integer, required
- `listing_id` - Integer, required

**Success Response (200):**
```json
{
  "success": true,
  "message": "Listing deleted successfully"
}
```

---

### Bump Listing
**POST** `/businesses/{business_id}/listings/{listing_id}/bump`

**Description:** Push listing to top of feed (like "refresh" on other platforms).

**Authentication:** Required (Bearer Token)

**URL Parameters:**
- `business_id` - Integer, required
- `listing_id` - Integer, required

**Success Response (200):**
```json
{
  "success": true,
  "message": "Listing bumped successfully",
  "data": {
    "bumped_at": "2025-12-21T10:00:00Z"
  }
}
```

---

### AI Generate Listing Content
**POST** `/listings/generate-content`

**Description:** Generate improved title, description, and tags using AI.

**Authentication:** Required (Bearer Token)

**Request Body:**
```json
{
  "title": "red dress",                  // Required, string, min:3, max:200
  "category": "Fashion",                 // Optional, string, max:100
  "additional_context": "size M, new"    // Optional, string, max:500
}
```

**Success Response (200):**
```json
{
  "success": true,
  "message": "Listing content generated successfully",
  "data": {
    "original_title": "red dress",
    "generated": {
      "title": "Stunning Red Evening Dress - Size M, Brand New with Tags",
      "description": "Elevate your wardrobe with this gorgeous red dress...\n\n**Key Features:**\n• Brand new with tags\n• Size M (fits true to size)\n• Vibrant red color\n• Premium quality fabric\n\nPerfect for parties, weddings, or date nights!",
      "tags": [
        "red dress",
        "evening dress",
        "party dress",
        "size M",
        "brand new",
        "women's fashion"
      ],
      "meta_title": "Red Evening Dress Size M - Brand New | Buy Now",
      "meta_description": "Stunning red dress, size M, brand new with tags. Perfect for parties and special occasions."
    }
  }
}
```

**Error Response (422):**
```json
{
  "success": false,
  "message": "Validation failed",
  "errors": {
    "title": ["The title must be at least 3 characters."]
  }
}
```

---

### Reverse Image Search
**POST** `/listings/search-by-image`

**Description:** Find similar listings by uploading a photo. Uses SHA-256 hash matching + Google Vision API.

**Authentication:** None (Public)

**Request Body (multipart/form-data):**
```json
{
  "image": "<file>",        // Required, image file, max:10MB, jpeg/png/jpg
  "limit": 20               // Optional, integer, 1-100, default:20
}
```

**Success Response (200):**
```json
{
  "success": true,
  "data": {
    "results": [
      {
        "listing": { /* full listing object */ },
        "similarity_score": 95.5,
        "match_type": "exact",  // "exact" or "similar"
        "matching_tags": ["red dress", "evening wear"]
      }
    ],
    "total_results": 5
  }
}
```

---

## Analytics & Tracking

### Track Impression
**POST** `/analytics/impressions`

**Description:** Track when listing appears in feed/search results. Call from frontend when listing enters viewport.

**Authentication:** None (Public)

**Request Body:**
```json
{
  "listing_id": 1,                       // Required, integer, exists in listings
  "source": "feed",                      // Optional, enum: feed|search|category|related|business_profile
  "context": {                           // Optional, object
    "position": 3,
    "search_query": "red dress",
    "filters": {"category": "Fashion"}
  }
}
```

**Success Response (200):**
```json
{
  "success": true,
  "data": {
    "tracked": true
  }
}
```

**Frontend Example:**
```javascript
// Track when listing appears in viewport
const observer = new IntersectionObserver((entries) => {
  entries.forEach(entry => {
    if (entry.isIntersecting) {
      fetch('/api/v1/analytics/impressions', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({
          listing_id: entry.target.dataset.listingId,
          source: 'feed',
          context: { position: entry.target.dataset.position }
        })
      });
    }
  });
}, { threshold: 0.5 });
```

---

### Track Click
**POST** `/analytics/clicks`

**Description:** Track when user clicks listing card. Call from frontend before navigation.

**Authentication:** None (Public)

**Request Body:**
```json
{
  "listing_id": 1,                       // Required, integer, exists in listings
  "source": "feed",                      // Optional, enum: feed|search|category|related
  "context": {                           // Optional, object
    "position": 3,
    "search_query": "red dress"
  }
}
```

**Success Response (200):**
```json
{
  "success": true,
  "data": {
    "tracked": true
  }
}
```

**Frontend Example:**
```javascript
function handleListingClick(listingId, slug) {
  // Track click
  fetch('/api/v1/analytics/clicks', {
    method: 'POST',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify({
      listing_id: listingId,
      source: 'feed'
    })
  });
  
  // Navigate
  window.location.href = `/listings/${slug}`;
}
```

---

### Track Interaction
**POST** `/analytics/interactions`

**Description:** Track user interactions (phone reveal, WhatsApp click, share, etc.).

**Authentication:** None (Public)

**Request Body:**
```json
{
  "listing_id": 1,                       // Required, integer, exists in listings
  "interaction_type": "phone_reveal",    // Required, enum: phone_reveal|whatsapp_click|email_click|share|favorite|report
  "metadata": {                          // Optional, object
    "platform": "facebook"
  }
}
```

**Success Response (200):**
```json
{
  "success": true,
  "data": {
    "tracked": true
  }
}
```

**Frontend Example:**
```javascript
// Phone reveal
async function revealPhone(listingId) {
  await fetch('/api/v1/analytics/interactions', {
    method: 'POST',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify({
      listing_id: listingId,
      interaction_type: 'phone_reveal'
    })
  });
  // Show phone number
}

// WhatsApp click
async function openWhatsApp(listingId, phone) {
  await fetch('/api/v1/analytics/interactions', {
    method: 'POST',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify({
      listing_id: listingId,
      interaction_type: 'whatsapp_click'
    })
  });
  window.open(`https://wa.me/${phone}`, '_blank');
}
```

---

### Get Listing Analytics
**GET** `/businesses/{business_id}/listings/{listing_id}/analytics`

**Description:** Get comprehensive analytics for a listing (owner/manager only).

**Authentication:** Required (Bearer Token)

**URL Parameters:**
- `business_id` - Integer, required
- `listing_id` - Integer, required

**Query Parameters:**
```
?period=30days  // Optional, enum: 7days|30days|90days|year, default:30days
```

**Success Response (200):**
```json
{
  "success": true,
  "data": {
    "overview": {
      "impressions": 5420,
      "clicks": 342,
      "views": 289,
      "unique_views": 245,
      "ctr": 6.31,
      "conversion_rate": 12.8
    },
    "engagement": {
      "phone_reveals": 37,
      "whatsapp_clicks": 28,
      "shares": 15
    },
    "period_stats": {
      "period": "30days",
      "impressions": 5420,
      "clicks": 342,
      "views": 289
    },
    "sources": {
      "impressions_by_source": {
        "feed": 3200,
        "search": 1800,
        "category": 420
      },
      "clicks_by_source": {
        "feed": 210,
        "search": 102,
        "category": 30
      }
    },
    "trends": {
      "daily_impressions": {
        "2025-12-01": 180,
        "2025-12-02": 195
      },
      "daily_clicks": {
        "2025-12-01": 12,
        "2025-12-02": 15
      }
    }
  }
}
```

---

### Get Platform Analytics
**GET** `/analytics/platform`

**Description:** Get platform-wide analytics (public for advertisers).

**Authentication:** None (Public)

**Query Parameters:**
```
?category=Fashion   // Optional, string - filter by category
&period=30days      // Optional, enum: 7days|30days|90days|year, default:30days
```

**Success Response (200):**
```json
{
  "success": true,
  "data": {
    "total_listings": 1250,
    "total_impressions": 2500000,
    "total_clicks": 87500,
    "total_views": 65000,
    "average_ctr": 3.5,
    "top_performing_listings": [
      {
        "id": 123,
        "title": "Red Evening Dress",
        "slug": "red-evening-dress-123",
        "impressions": 15000,
        "clicks": 890,
        "views": 750,
        "ctr": 5.93
      }
    ],
    "category_breakdown": [
      {
        "category": "Fashion",
        "listings": 450,
        "impressions": 980000,
        "clicks": 35000,
        "ctr": 3.57
      }
    ]
  }
}
```

---

### Get View Statistics
**GET** `/businesses/{business_id}/listings/{listing_id}/views/stats`

**Description:** Get view statistics for a listing.

**Authentication:** Required (Bearer Token)

**URL Parameters:**
- `business_id` - Integer, required
- `listing_id` - Integer, required

**Success Response (200):**
```json
{
  "success": true,
  "data": {
    "total_views": 1250,
    "unique_views": 890,
    "views_today": 45,
    "views_this_week": 320,
    "views_this_month": 1100
  }
}
```

---

### Get Viewers List
**GET** `/businesses/{business_id}/listings/{listing_id}/views/viewers`

**Description:** Get list of who viewed the listing.

**Authentication:** Required (Bearer Token)

**URL Parameters:**
- `business_id` - Integer, required
- `listing_id` - Integer, required

**Success Response (200):**
```json
{
  "success": true,
  "data": {
    "viewers": [
      {
        "id": 123,
        "user": {
          "id": 45,
          "name": "John Doe",
          "email": "john@example.com"
        },
        "ip_address": null,
        "viewed_at": "2025-12-21T06:30:00Z",
        "is_authenticated": true
      },
      {
        "id": 124,
        "user": null,
        "ip_address": "192.168.xxx.xxx",
        "viewed_at": "2025-12-21T06:25:00Z",
        "is_authenticated": false
      }
    ],
    "total_count": 50
  }
}
```

---

## Feedback

### Submit Feedback
**POST** `/businesses/{business_id}/feedback`

**Description:** Submit feedback/review for a business.

**Authentication:** Required (Bearer Token)

**URL Parameters:**
- `business_id` - Integer, required

**Request Body (multipart/form-data):**
```json
{
  "rating": 5,                           // Required, integer, 1-5
  "comment": "Great service!",           // Required, string, min:10
  "attachments": ["<file1>", "<file2>"]  // Optional, array of files, max:5, each max:5MB
}
```

**Success Response (201):**
```json
{
  "success": true,
  "message": "Feedback submitted successfully. It will be reviewed before publishing.",
  "data": {
    "id": 1,
    "rating": 5,
    "comment": "Great service!",
    "status": "pending",
    "created_at": "2025-12-21T06:00:00Z"
  }
}
```

---

### Get Business Feedback (Public)
**GET** `/businesses/{slug}/feedback`

**Description:** Get approved feedback for a business (public).

**Authentication:** None (Public)

**URL Parameters:**
- `slug` - String, required

**Query Parameters:**
```
?per_page=20  // Optional, integer, 1-100, default:20
&page=1       // Optional, integer, default:1
```

**Success Response (200):**
```json
{
  "success": true,
  "data": {
    "feedback": [
      {
        "id": 1,
        "rating": 5,
        "comment": "Great service!",
        "user": {
          "id": 1,
          "name": "John Doe"
        },
        "response": {
          "comment": "Thank you!",
          "responded_at": "2025-12-21T07:00:00Z"
        },
        "created_at": "2025-12-21T06:00:00Z"
      }
    ],
    "pagination": {
      "current_page": 1,
      "per_page": 20,
      "total": 50,
      "last_page": 3
    },
    "summary": {
      "average_rating": 4.5,
      "total_feedback": 50,
      "rating_distribution": {
        "5": 30,
        "4": 15,
        "3": 3,
        "2": 1,
        "1": 1
      }
    }
  }
}
```

---

## Admin

### List Users (Admin)
**GET** `/admin/users`

**Description:** List all users with filters (admin only).

**Authentication:** Required (Bearer Token, Admin role)

**Query Parameters:**
```
?status=active     // Optional, enum: active|suspended|deleted
&role=seller       // Optional, enum: buyer|seller|admin
&search=john       // Optional, string - search in name/email
&per_page=20       // Optional, integer, 1-100, default:20
&page=1            // Optional, integer, default:1
```

**Success Response (200):**
```json
{
  "success": true,
  "data": {
    "users": [ /* array of user objects */ ],
    "pagination": { /* pagination object */ }
  }
}
```

---

## Error Responses

### Common Error Codes

**400 Bad Request:**
```json
{
  "success": false,
  "message": "Invalid request format"
}
```

**401 Unauthorized:**
```json
{
  "success": false,
  "message": "Unauthenticated. Please login."
}
```

**403 Forbidden:**
```json
{
  "success": false,
  "message": "You do not have permission to perform this action"
}
```

**404 Not Found:**
```json
{
  "success": false,
  "message": "Resource not found"
}
```

**422 Validation Error:**
```json
{
  "success": false,
  "message": "Validation failed",
  "errors": {
    "field_name": ["Error message 1", "Error message 2"]
  }
}
```

**429 Too Many Requests:**
```json
{
  "success": false,
  "message": "Too many requests. Please try again later."
}
```

**500 Internal Server Error:**
```json
{
  "success": false,
  "message": "An error occurred. Please try again later."
}
```

---

## Rate Limiting

- **Authentication endpoints:** 5 requests per minute
- **General API:** 60 requests per minute
- **File uploads:** 10 requests per minute

---

## File Upload Limits

- **Profile photos:** Max 5MB, jpeg/png/jpg
- **Business logos:** Max 5MB, jpeg/png/jpg
- **Business covers:** Max 10MB, jpeg/png/jpg
- **Listing media:** Max 10MB per file, jpeg/png/jpg/mp4, max 10 files
- **Feedback attachments:** Max 5MB per file, max 5 files

---

## Notes

1. **All timestamps** are in ISO 8601 format (UTC): `2025-12-21T06:00:00Z`
2. **All prices** are returned as strings with 2 decimal places: `"25000.00"`
3. **Phone numbers** must be in E.164 format: `+256700123456`
4. **Pagination** is available on all list endpoints
5. **Soft deletes** are used - deleted items can be restored by admins
6. **View tracking** is automatic on listing detail page
7. **Impression/click tracking** requires frontend implementation
8. **File uploads** use `multipart/form-data` content type
9. **JWT tokens** expire after 1 hour (3600 seconds)
10. **Refresh tokens** are valid for 30 days

---

**Last Updated:** December 21, 2025  
**API Version:** v1  
**Support:** support@yilick.com
